package com.bxm.adx.common.sell;

import com.bxm.adx.common.buy.Buyer;
import com.bxm.adx.common.buy.buyers.PriorityBuyers;
import com.bxm.adx.common.market.MarketOrders;
import com.bxm.adx.common.market.MarketRequest;
import com.bxm.adx.common.sell.position.Position;
import com.bxm.adx.common.sell.position.PositionService;
import com.bxm.adx.common.sell.request.Impression;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.lang.StringUtils;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.CollectionUtils;

import java.util.*;

/**
 * @author allen
 * @since 2019-12-20
 */
@Configuration
public class DefaultBidRequestWrapper implements BidRequestWrapper {

    private final PriorityBuyers priorityBuyers;
    private final PositionService positionService;

    public DefaultBidRequestWrapper(PriorityBuyers priorityBuyers, PositionService positionService) {
        this.priorityBuyers = priorityBuyers;
        this.positionService = positionService;
    }

    @Override
    public MarketOrders packing(BidRequest bidRequest) {
        if (null == bidRequest) {
            return null;
        }
        List<Impression> imps = bidRequest.getImps();
        if (CollectionUtils.isEmpty(imps)) {
            return null;
        }

        Map<Position, List<List<Buyer>>> positionListMap = Maps.newHashMap();
        List<Position> positionList = new ArrayList<>();
        // 获取所有的广告位
        for (Impression imp : imps) {
            String tag_id = imp.getTag_id();
            if (StringUtils.isBlank(tag_id)) {
                continue;
            }
            Position position = positionService.getByPositionId(tag_id);
            if(null != position){
                positionList.add(position);
            }
            if (null != position && position.isEnabled()) {
                // 查找这些广告位分配的卖家和等级
                List<List<Buyer>> buyers = priorityBuyers.findAsPriority(position);
                positionListMap.put(position, buyers);
            }
        }

        // 索引即表示优先级，从0开始。
        List<MarketRequest> requests = Lists.newArrayListWithCapacity(5);
        // 重组，分出每一个优先级中对应的DSP+广告位
        for (Map.Entry<Position, List<List<Buyer>>> positionListEntry : positionListMap.entrySet()) {
            Position position = positionListEntry.getKey();
            List<List<Buyer>> list = positionListEntry.getValue();
            int level = list.size();
            if (level == 0) {
                // 没有设置分流策略
                continue;
            }

            for (int i = 0; i < level; i++) {
                // 这个广告位第i级的DSP配置列表
                List<Buyer> array = list.get(i);
                if (CollectionUtils.isEmpty(array)) {
                    continue;
                }
                MarketRequest marketRequest = CollectionUtils.isEmpty(requests) ? null : requests.size() > i ? requests.get(i) : null;
                if (null == marketRequest) {
                    marketRequest = new MarketRequest(i);
                    requests.add(i, marketRequest);
                }
                marketRequest.addBuyers(array);
                marketRequest.addPosition(position);
            }
        }

        // 合并，按照DSP进行分组，同一个DSP在不同优先级时取最高优先级，并将广告位进行合并。
        for (int i = 0; i < requests.size(); i++) {
            MarketRequest marketRequest = requests.get(i);
            Set<Buyer> buyers = marketRequest.getBuyers();
            Set<Position> positions = marketRequest.getPositions();
            // 判断小于 i 级是否有相同的DSP
            for (int j = i + 1; j < requests.size(); j++) {
                MarketRequest low = requests.get(j);
                Set<Buyer> lowBuyers = low.getBuyers();
                Set<Position> lowPositions = low.getPositions();
                Iterator<Buyer> iterator = lowBuyers.iterator();
                while (iterator.hasNext()) {
                    if (buyers.contains(iterator.next())) {
                        // 这个低级有相同DSP，那么将广告位赋予给它，并且删除这个低级的DSP
                        positions.addAll(lowPositions);
                        iterator.remove();
                    }
                }
            }
        }
        return new MarketOrders(bidRequest, requests, positionList);
    }
}
