package com.bxm.adx.common.market.exchange;

import com.bxm.adx.common.buy.Buyer;
import com.bxm.adx.common.sell.BidRequest;
import com.google.common.collect.Maps;

import java.util.Map;

/**
 * 买方交易上下文，线程独享。
 * 可以通过 {@link #put(Object, Object)} 来设置一些内容，然后再使用 {@link #get(Object)} 来获取。
 * @author allen
 * @date 2020-11-05
 * @since 1.0
 */
public class ExchangeContext {

    private static final ThreadLocal<ExchangeContext> THREAD_LOCAL = new ThreadLocal<ExchangeContext>() {
        @Override
        protected ExchangeContext initialValue() {
            return new ExchangeContext();
        }
    };

    private static final String KEY_BIDREQUEST = "$$BID_REQUEST$$";
    private static final String KEY_BUYER = "$$BUYER$$";

    private final Map<Object, Object> attachment = Maps.newHashMap();

    private static ExchangeContext get() {
        return THREAD_LOCAL.get();
    }

    public static void remove() {
        THREAD_LOCAL.remove();
    }

    static void putBidRequest(BidRequest bidRequest) {
        get().attachment.put(KEY_BIDREQUEST, bidRequest);
    }

    static void putBuyer(Buyer buyer) {
        get().attachment.put(KEY_BUYER, buyer);
    }

    public static void put(Object key, Object value) {
        ExchangeContext context = get();
        context.attachment.put(key, value);
    }

    public static Object get(Object key) {
        ExchangeContext context = get();
        return context.attachment.get(key);
    }

    public static BidRequest getBidRequest() {
        Object o = get().attachment.get(KEY_BIDREQUEST);
        if (o instanceof BidRequest) {
            return (BidRequest) o;
        }
        return null;
    }

    public static Buyer getBuyer() {
        Object o = get().attachment.get(KEY_BUYER);
        if (o instanceof Buyer) {
            return (Buyer) o;
        }
        return null;
    }
}
