package com.bxm.adx.common.market.exchange;

import com.bxm.adx.common.AdxProperties;
import com.bxm.adx.common.adapter.AdxContext;
import com.bxm.adx.common.adapter.AdxContextFactory;
import com.bxm.adx.common.buy.buyers.BuyerWrapper;
import com.bxm.adx.common.market.Deal;
import com.bxm.adx.common.market.MarketOrders;
import com.bxm.adx.common.market.MarketRequest;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.position.Position;
import com.bxm.adx.common.sell.request.Impression;
import com.bxm.adx.facade.constant.enums.AdxErrEnum;
import com.bxm.adx.facade.exception.AdxException;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.context.annotation.Configuration;

import java.util.Collection;
import java.util.List;
import java.util.Set;

/**
 * @author allen
 * @since 2019-12-18
 */
@Slf4j
@Configuration
public class DefaultPriorityExchanger implements PriorityExchanger {

    private final AdxProperties properties;
    private final Exchanger exchanger;

    public DefaultPriorityExchanger(AdxProperties properties, Exchanger exchanger) {
        this.properties = properties;
        this.exchanger = exchanger;
    }

    @Override
    public List<Deal> exchange(MarketOrders orders) {
        BidRequest request = orders.getBidRequest();
        List<MarketRequest> requestList = orders.getRequests();
        if (CollectionUtils.isEmpty(requestList)) {
            return null;
        }
        //成功的dsp交易结果集合
        List<Deal> result = Lists.newArrayList();
        //失败的dsp交易结果集合
        List<Deal> failResult = Lists.newArrayList();

        for (MarketRequest marketRequest : requestList) {
            Set<BuyerWrapper> buyers = marketRequest.getBuyers();
            Set<Position> positions = marketRequest.getPositions();
            Set<String> positionIds = Sets.newHashSet();
            for (Position position : positions) {
                positionIds.add(position.getPositionId());
            }
            List<Impression> imps = Lists.newArrayList(request.getImps());

            imps.removeIf(impression -> !positionIds.contains(impression.getTag_id()));
            final BidRequest e = new BidRequest();
            BeanUtils.copyProperties(request, e);
            e.setImps(imps);

            List<Deal> deals = exchanger.bidding(e, buyers);
            if (CollectionUtils.isNotEmpty(deals)) {
                for (Deal deal : deals) {
                    if (deal.isBidSuccess()) {
                        result.add(deal);
                    } else {
                        failResult.add(deal);
                    }
                }
            }
            // 如果当前有竞价结果，那么直接返回
            if (!AdxContextFactory.get().isOptimization()) {
                if (CollectionUtils.isNotEmpty(result)) {
                    return result;
                }
            }
        }

        if (CollectionUtils.isNotEmpty(result)) {
            return result;
        }

        if (failResult.size() == 1) {
            throw new AdxException(failResult.iterator().next().getAdxErrEnum());
        } else {
            //有多个返回错误的情况下统一返回dsp请求超时
            throw new AdxException(AdxErrEnum.DSP_EMPTY_RESPONSE);
        }
//        return result;
    }
}
