package com.bxm.adx.common.sdkconfig;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.context.annotation.Configuration;

import com.bxm.adx.facade.constant.enums.AdxErrEnum;
import com.bxm.adx.facade.exception.AdxException;
import com.bxm.adx.common.PositionSdkConfigChannelEnum;
import com.bxm.adx.common.ingetration.UserServiceIntegration;
import com.bxm.adx.common.sell.position.Position;
import com.bxm.adx.common.sell.position.PositionService;
import com.bxm.adx.facade.constant.enums.DevOsEnum;
import com.bxm.adx.facade.constant.enums.SdkVersionEnum;
import com.bxm.adx.facade.model.position.SdkConfigRequest;
import com.bxm.adx.facade.model.position.SdkConfigResponse;
import com.bxm.user.facade.DevRequest;
import com.bxm.user.facade.UserProfile;
import com.bxm.warcar.utils.DateHelper;
import com.bxm.warcar.utils.StringHelper;

import lombok.extern.slf4j.Slf4j;

/**
 * <p>
 * 广告位SDK配置服务实现
 * </p>
 *
 * @author zhengwangeng
 * @since 2020-1-7
 */
@Slf4j
@Configuration
public class SdkConfigServiceImpl implements SdkConfigService {
    
    private final SdkConfigDao sdkConfigDao;
    private final PositionService positionService;
    private final UserServiceIntegration userService;

    public SdkConfigServiceImpl(SdkConfigDao sdkConfigDao, PositionService positionService,
                                UserServiceIntegration userService) {
        this.sdkConfigDao = sdkConfigDao;
        this.positionService = positionService;
        this.userService = userService;
    }

    @Override
    public List<SdkConfigResponse> getByPositionId(SdkConfigRequest sdkConfigRequest) {
        //3.2.5版本开始需要校验
        boolean isSupportException = StringUtils.isNotBlank(sdkConfigRequest.getSdkv())
                && StringHelper.isNewVersion(SdkVersionEnum.SUPPORT_EXCEPTION_325.getVersion(), sdkConfigRequest.getSdkv());

        String positionId = sdkConfigRequest.getPositionId();
        Position position = positionService.getByPositionId(positionId);
        if (null == position) {
            if (log.isDebugEnabled()) {
                log.debug("广告位SDK配置中，没有获取到：{} 的广告位信息。", positionId);
            }
            if (isSupportException) {
                throw new AdxException(AdxErrEnum.POSITION_NOT_FOUND);
            } else {
                return Collections.emptyList();
            }
        }
        if (!position.getPositionScene().equals(sdkConfigRequest.getPositionScene())) {
            if (log.isDebugEnabled()) {
                log.debug("广告位SDK配置中，广告位：{} 的场景信息不匹配。", positionId);
            }
            if (isSupportException) {
                throw new AdxException(AdxErrEnum.POSITION_SCENE_MISMATCH);
            } else {
                return Collections.emptyList();
            }
        }

        if (isSupportException) {
            //验证广告位的正确性（三码合一）
            boolean result = positionService.verifyPosition(position, sdkConfigRequest.getAppId(),
                    sdkConfigRequest.getPackageName(), sdkConfigRequest.getSdkv());
            if (!result) {
                throw new AdxException(AdxErrEnum.APP_INFO_ERR);
            }
        }

        List<SdkConfig> sdkConfigs = sdkConfigDao.getByPositionId(positionId);
        if(Boolean.TRUE.equals(sdkConfigRequest.getIsDebug())){
            sdkConfigs = getSdkConfigsForDebug(sdkConfigs);
        }else{
            sdkConfigs = getSdkConfigForDynamic(sdkConfigs, sdkConfigRequest);
        }
        List<SdkConfigResponse> sdkConfigResponses = new ArrayList<>(sdkConfigs.size());
        for (SdkConfig sdkConfig : sdkConfigs) {
            SdkConfigResponse sdkConfigResponse = new SdkConfigResponse();
            BeanUtils.copyProperties(sdkConfig, sdkConfigResponse);
            sdkConfigResponses.add(sdkConfigResponse);
        }
        return sdkConfigResponses;
    }

    private List<SdkConfig> getSdkConfigForDynamic(List<SdkConfig> sdkConfigs, SdkConfigRequest sdkConfigRequest) {
        UserProfile user = userService.getUser(DevRequest.builder()
                .os(sdkConfigRequest.getOs())
                .imei(sdkConfigRequest.getImei())
                .imeiMd5(sdkConfigRequest.getImei_md5())
                .anid(sdkConfigRequest.getAnid())
                .anidMd5(sdkConfigRequest.getAnid_md5())
                .oaid(sdkConfigRequest.getOaid())
                .idfa(sdkConfigRequest.getIdfa())
                .idfaMd5(sdkConfigRequest.getIdfa_md5())
                .build());
        if(null == user || StringUtils.isBlank(user.getUid())){
            return sdkConfigs;
        }
        String devNo = user.getUid();

        SdkConfig bxmBackupSdkConfig = null;
        SdkConfig bdSdkConfig = null;
        SdkConfig ksSdkConfig = null;
        SdkConfig adSdkConfig = null;

        boolean isIos = Objects.equals(sdkConfigRequest.getOs(), DevOsEnum.IOS.getId());
        boolean isSupportBdAndKsAndAd = StringUtils.isNotBlank(sdkConfigRequest.getSdkv())
                && StringHelper.isNewVersion(SdkVersionEnum.IOS_SUPPORT_BD_AND_KS_331.getVersion(), sdkConfigRequest.getSdkv());
        for (SdkConfig conf : sdkConfigs) {
            if(Objects.equals(PositionSdkConfigChannelEnum.BACKUP.getCode(), conf.getChannelType())){
                bxmBackupSdkConfig = conf;
            }
            if(isIos && isSupportBdAndKsAndAd == false){
                if(Objects.equals(PositionSdkConfigChannelEnum.BD.getCode(), conf.getChannelType())){
                    bdSdkConfig = conf;
                }else if(Objects.equals(PositionSdkConfigChannelEnum.KS.getCode(), conf.getChannelType())){
                    ksSdkConfig = conf;
                }else if(Objects.equals(PositionSdkConfigChannelEnum.AD.getCode(), conf.getChannelType())){
                    adSdkConfig = conf;
                }
            }

        }
        if (bxmBackupSdkConfig != null) {
            //先移除，后面再加
            sdkConfigs.remove(bxmBackupSdkConfig);
        }

        //如果版本不支持，百度和快手不返回
        if (bdSdkConfig != null) {
            sdkConfigs.remove(bdSdkConfig);
        }
        if (ksSdkConfig != null) {
            sdkConfigs.remove(ksSdkConfig);
        }
        if (adSdkConfig != null) {
            sdkConfigs.remove(adSdkConfig);
        }

        //移除流量占比为0的
        sdkConfigs.removeIf(
                p -> (p.getRateOfFlow() != null && p.getRateOfFlow() == 0)
        );

        String dmy = DateHelper.format(new Date(), DateHelper.PATTERN_STR8);
        Map<String, Integer> channelTypeAndViewNumMap =
                sdkConfigDao.getByPositionIdDateChannelViewData(sdkConfigRequest.getPositionId(), dmy, devNo);

        boolean needRateFilter = Objects.nonNull(sdkConfigs.stream().filter(p ->
                (p.getRateOfFlow() != null && p.getRateOfFlow() != 100)).findFirst().orElse(null));
        if (needRateFilter) {
            rateAndFrequencyControlFilter(sdkConfigs, sdkConfigRequest, devNo, dmy, channelTypeAndViewNumMap);
        } else {
            frequencyControlFilter(sdkConfigs, sdkConfigRequest, devNo, dmy, channelTypeAndViewNumMap);
        }

        boolean isSupportBxmBackup = StringUtils.isNotBlank(sdkConfigRequest.getSdkv())
                && StringHelper.isNewVersion(SdkVersionEnum.SUPPORT_BXM_BACKUP_30.getVersion(), sdkConfigRequest.getSdkv());
        if (isSupportBxmBackup && bxmBackupSdkConfig != null) {
            //前面移除，这里再加上
            sdkConfigs.add(bxmBackupSdkConfig);
        }

        //这里为了节约服务器资源就不重新排序了，由前端sdk排序取最小的sort值
        //Collections.sort(sdkConfigs, (p1, p2) -> p1.getSort().compareTo(p2.getSort()));
        return sdkConfigs;
    }

    /**
     * 流量占比和频次控制过滤，重新排序sdk渠道
     * @param sdkConfigs
     * @param sdkConfigRequest
     * @param devNo
     * @param dmy
     * @param channelTypeAndViewNumMap
     */
    private void rateAndFrequencyControlFilter(List<SdkConfig> sdkConfigs, SdkConfigRequest sdkConfigRequest, String devNo,
                                               String dmy, Map<String, Integer> channelTypeAndViewNumMap) {
        Map<String, Integer> testDevMap = getTestMobileDev();
        int frequencyControlLoopCount = 0;
        SdkConfig lastSdkConfig = sdkConfigs.get(sdkConfigs.size() - 1);
        int lastSort = lastSdkConfig.getSort();
        for (SdkConfig sdkConfig : sdkConfigs) {
            if (sdkConfig.getRateOfFlow() != null && sdkConfig.getRateOfFlow() != 100) {
                //设置了流量占比，需要满足占比，然后再判断频次控制
                boolean isTopBxm = Math.abs(devNo.hashCode() % 100) < testDevMap.getOrDefault(devNo, sdkConfig.getRateOfFlow()) ? true : false;
                if (isTopBxm) {
                    if (channelTypeAndViewNumMap != null && channelTypeAndViewNumMap.size() > 0) {
                        Integer viewNum = channelTypeAndViewNumMap.get(PositionSdkConfigChannelEnum.get(sdkConfig.getChannelType()).getDateChannelView());
                        if (viewNum != null && sdkConfig.getFrequency() != null && viewNum >= sdkConfig.getFrequency()) {
                            lastSort = lastSort + 1;
                            sdkConfig.setSort(lastSort);
                            frequencyControlLoopCount++;
                        }
                    }
                }else{
                    lastSort = lastSort + 1;
                    sdkConfig.setSort(lastSort);
                }
                continue;
            }
            //直接进行渠道的频次控制
            if (channelTypeAndViewNumMap != null && channelTypeAndViewNumMap.size() > 0) {
                Integer viewNum = channelTypeAndViewNumMap.get(PositionSdkConfigChannelEnum.get(sdkConfig.getChannelType()).getDateChannelView());
                if (viewNum != null && sdkConfig.getFrequency() != null && viewNum >= sdkConfig.getFrequency()) {
                    lastSort = lastSort + 1;
                    sdkConfig.setSort(lastSort);
                    frequencyControlLoopCount++;
                }
            }
        }

        if (frequencyControlLoopCount >= sdkConfigs.size()) {
            //清除之前的频次数据
            sdkConfigDao.deletePositionIdDateChannelViewData(sdkConfigRequest.getPositionId(), dmy, devNo);
        }
    }

    /**
     * 频次控制过滤,重新排序sdk渠道
     * @param sdkConfigs
     * @param sdkConfigRequest
     * @param devNo
     */
    private void frequencyControlFilter(List<SdkConfig> sdkConfigs, SdkConfigRequest sdkConfigRequest, String devNo,
                                        String dmy, Map<String, Integer> channelTypeAndViewNumMap) {
        if (channelTypeAndViewNumMap != null && channelTypeAndViewNumMap.size() > 0) {
            int frequencyControlLoopCount = 0;
            SdkConfig lastSdkConfig = sdkConfigs.get(sdkConfigs.size() - 1);
            int lastSort = lastSdkConfig.getSort();
            //频次控制判断，重新调整顺序
            for (SdkConfig sdkConfig : sdkConfigs) {
                Integer viewNum = channelTypeAndViewNumMap.get(PositionSdkConfigChannelEnum.get(sdkConfig.getChannelType()).getDateChannelView());
                if (viewNum != null && sdkConfig.getFrequency() != null && viewNum >= sdkConfig.getFrequency()) {
                    lastSort = lastSort + 1;
                    sdkConfig.setSort(lastSort);
                    frequencyControlLoopCount++;
                }
            }
            if (frequencyControlLoopCount >= sdkConfigs.size()) {
                //清除之前的频次数据
                sdkConfigDao.deletePositionIdDateChannelViewData(sdkConfigRequest.getPositionId(), dmy, devNo);
            }
        }
    }

    /**
     * 测试的手机设备
     * @return
     */
    private Map<String, Integer> getTestMobileDev() {
        Map<String, Integer> testDevMap = new HashMap(2);

        //iphone6（黑色） ：  F472E98E-32BE-46AD-AB49-78C0C6FA1868    正常流量
        //testDevMap.put("F472E98E-32BE-46AD-AB49-78C0C6FA1868", bxmSdkConfig.getRateOfFlow());
        //iphone6P（金色） ： FD4AA4C3-8859-4CB4-A9B5-37424FFB4EFC    100%命中
        testDevMap.put("FD4AA4C3-8859-4CB4-A9B5-37424FFB4EFC", 100);


        //魅族		867648032168948     正常流量
        //testDevMap.put("867648032168948", bxmSdkConfig.getRateOfFlow());
        //mi6x	868133036330918     100%命中
        testDevMap.put("868133036330918", 100);

        //粉色红米
        //testDevMap.put("867251037003788", 100);
        return testDevMap;
    }

    /**
     * 获取debug数据
     * 只获取变现猫渠道的数据
     */
    private List<SdkConfig> getSdkConfigsForDebug(List<SdkConfig> sdkConfigs) {
        if(CollectionUtils.isEmpty(sdkConfigs)){
            return Collections.emptyList();
        }
        List<SdkConfig> resps = new ArrayList<>();
        for (SdkConfig conf : sdkConfigs) {
            if(Objects.equals(PositionSdkConfigChannelEnum.BXM.getCode(), conf.getChannelType())){
                resps.add(conf);
                return resps;
            }

        }
        return resps;
    }
}
