package com.bxm.adx.common.sell;

import com.bxm.adx.common.adapter.AdxContextFactory;
import com.bxm.adx.common.buy.dispatcher.Dispatcher;
import com.bxm.adx.common.buy.optimization.Optimization;
import com.bxm.adx.common.buy.optimization.OptimizationDao;
import com.bxm.adx.common.ip.IpService;
import com.bxm.adx.facade.constant.enums.AdxErrEnum;
import com.bxm.adx.facade.exception.AdxException;
import com.bxm.adx.common.AdxProperties;
import com.bxm.adx.common.buy.Buyer;
import com.bxm.adx.common.buy.buyers.BuyerWrapper;
import com.bxm.adx.common.buy.buyers.PriorityBuyers;
import com.bxm.adx.common.ingetration.UserServiceIntegration;
import com.bxm.adx.common.market.MarketOrders;
import com.bxm.adx.common.market.MarketRequest;
import com.bxm.adx.common.sell.position.Position;
import com.bxm.adx.common.sell.position.PositionService;
import com.bxm.adx.common.sell.request.Device;
import com.bxm.adx.common.sell.request.Impression;
import com.bxm.user.facade.DevRequest;
import com.bxm.user.facade.UserProfile;
import com.bxm.warcar.ip.IP;
import com.bxm.warcar.utils.JsonHelper;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.CollectionUtils;

import java.util.*;

/**
 * @author allen
 * @since 2019-12-20
 */
@Slf4j
@Configuration
public class DefaultBidRequestWrapper implements BidRequestWrapper {

    private final PriorityBuyers priorityBuyers;
    private final PositionService positionService;
    private final AdxProperties properties;
    private final UserServiceIntegration userServiceIntegration;
    private final OptimizationDao optimizationDao;
    private final IpService ipService;

    public DefaultBidRequestWrapper(PriorityBuyers priorityBuyers, PositionService positionService,
                                    AdxProperties properties, UserServiceIntegration userServiceIntegration,
                                    OptimizationDao optimizationDao, IpService ipService) {
        this.priorityBuyers = priorityBuyers;
        this.positionService = positionService;
        this.properties = properties;
        this.userServiceIntegration = userServiceIntegration;
        this.optimizationDao = optimizationDao;
        this.ipService = ipService;
    }

    @Override
    public MarketOrders packing(BidRequest bidRequest) {
        if (null == bidRequest) {
            return null;
        }
        AdxContextFactory.get().setBidRequest(bidRequest);
        List<Impression> imps = bidRequest.getImps();
        if (CollectionUtils.isEmpty(imps)) {
            throw new AdxException(AdxErrEnum.POSITION_NOT_FOUND);
        }

        Map<Position, List<List<BuyerWrapper>>> positionListMap = Maps.newHashMap();
        List<Position> positionList = new ArrayList<>();
        // 获取所有的广告位
        for (Impression imp : imps) {
            String tag_id = imp.getTag_id();
            if (StringUtils.isBlank(tag_id)) {
                continue;
            }
            Position position = positionService.getByPositionId(tag_id);
            if (null != position) {
                positionList.add(position);
            }
            if (null != position && position.isEnabled()) {
                // 查找这些广告位分配的卖家和等级
                List<List<BuyerWrapper>> buyers = priorityBuyers.findAsPriority(position, imp.getBid_floor(), imp.getBid_top());
                //处理广告优化
                handleOptimization(position, bidRequest, buyers);
                handleBackupEntrance(bidRequest, buyers);
                positionListMap.put(position, buyers);
            }
        }

        if (CollectionUtils.isEmpty(positionListMap)) {
            throw new AdxException(AdxErrEnum.POSITION_NOT_FOUND);
        }

        // 索引即表示优先级，从0开始。
        List<MarketRequest> requests = Lists.newArrayListWithCapacity(properties.getMaxCapacityOfBuyers());
        // 重组，分出每一个优先级中对应的DSP+广告位
        for (Map.Entry<Position, List<List<BuyerWrapper>>> positionListEntry : positionListMap.entrySet()) {
            Position position = positionListEntry.getKey();
            List<List<BuyerWrapper>> list = positionListEntry.getValue();
            int level = list.size();
            if (level == 0) {
                // 没有设置分流策略
                continue;
            }

            for (int i = 0; i < level; i++) {
                // 这个广告位第i级的DSP配置列表
                List<BuyerWrapper> array = list.get(i);
                if (CollectionUtils.isEmpty(array)) {
                    continue;
                }
                MarketRequest marketRequest = CollectionUtils.isEmpty(requests) ? null : requests.size() > i ? requests.get(i) : null;
                if (null == marketRequest) {
                    marketRequest = new MarketRequest(i);
                    requests.add(i, marketRequest);
                }
                marketRequest.addBuyers(array);
                marketRequest.addPosition(position);
            }
        }

//        // 合并，按照DSP进行分组，同一个DSP在不同优先级时取最高优先级，并将广告位进行合并。
//        for (int i = 0; i < requests.size(); i++) {
//            MarketRequest marketRequest = requests.get(i);
//            Set<Buyer> buyers = marketRequest.getBuyers();
//            Set<Position> positions = marketRequest.getPositions();
//            // 判断小于 i 级是否有相同的DSP
//            for (int j = i + 1; j < requests.size(); j++) {
//                MarketRequest low = requests.get(j);
//                Set<Buyer> lowBuyers = low.getBuyers();
//                Set<Position> lowPositions = low.getPositions();
//                Iterator<Buyer> iterator = lowBuyers.iterator();
//                while (iterator.hasNext()) {
//                    if (buyers.contains(iterator.next())) {
//                        // 这个低级有相同DSP，那么将广告位赋予给它，并且删除这个低级的DSP
//                        positions.addAll(lowPositions);
//                        iterator.remove();
//                    }
//                }
//            }
//        }

        MarketOrders marketOrders = new MarketOrders(bidRequest, requests, positionList, getUid(bidRequest));
        if (log.isDebugEnabled()) {
            log.debug("{}", JsonHelper.convert(marketOrders));
        }
        return marketOrders;
    }

    private void handleBackupEntrance(BidRequest request, List<List<BuyerWrapper>> buyers) {
        String entanceName = request.getEntrance_name();
        if (StringUtils.isEmpty(entanceName)) {
            return;
        }
        if ("backup".equalsIgnoreCase(entanceName)) {
            buyers.clear();
            List<BuyerWrapper> buyerList = new ArrayList<>();
            buyers.add(buyerList);
            Map<String, Buyer> allBuyers = priorityBuyers.getAllBuyers();
            Buyer buyer = allBuyers.get("pangu-backup");
            if (buyer == null) {
                throw new AdxException("can't find backup-dsp ");
            }
            buyerList.add(new BuyerWrapper(buyer, null));
        }
    }

    private void handleOptimization(Position position, BidRequest request, List<List<BuyerWrapper>> buyers) {
        Optimization optimization = optimizationDao.getOptimizationByPositionId(position.getPositionId());
        if (Objects.isNull(optimization)) {
            return;
        }
        String regionCodes = optimization.getRegion();
        if (!org.springframework.util.StringUtils.isEmpty(regionCodes)) {
            IP ipp = getIp(request);
            Set<String> set = Sets.newHashSet();
            set.addAll(Arrays.asList(regionCodes.split(",")));
            if (Objects.isNull(ipp) || !ipp.in(set)) {
                return;
            }
        }

        Map<String, Buyer> allBuyers = priorityBuyers.getAllBuyers();
        Buyer buyer = allBuyers.get("optimization");
        if (buyer == null) {
            throw new AdxException("can't find op-dsp ");
        }
        Dispatcher dispatcher = new Dispatcher();
        dispatcher.setDspPosid(optimization.getDspPositionId());
        buyers.add(Lists.newArrayList(new BuyerWrapper(buyer, dispatcher)));
        AdxContextFactory.get().setOptimization(true);
    }

    /**
     * 统一方法获取uid
     *
     * @param bidRequest
     * @return
     */
    private String getUid(BidRequest bidRequest) {

        Device device = bidRequest.getDevice();
        if (device == null) {
            return null;
        }

        final DevRequest devRequest = DevRequest.builder()
                .oaid(device.getOaid())
                .imeiMd5(device.getImei_md5())
                .imei(device.getImei())
                .idfaMd5(device.getIdfa_md5())
                .idfa(device.getIdfa())
                .anidMd5(device.getDpid_md5())
                .anid(device.getDpid())
                .build();
        UserProfile user = userServiceIntegration.getUser(devRequest);
        if (null == user) {
            return null;
        }
        return user.getUid();
    }

    private IP getIp(BidRequest bidRequest) {
        IP ipp = null;
        if (bidRequest != null) {
            Device device = bidRequest.getDevice();
            if (device != null) {
                String ip = device.getIp();
                if (!org.springframework.util.StringUtils.isEmpty(ip)) {
                    ipp = ipService.analyze(ip);
                    AdxContextFactory.get().setIp(ipp);
                }
            }
        }
        return ipp;
    }
}
