package com.bxm.adx.common.filter;

import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.market.MarketOrders;
import com.bxm.adx.common.sell.position.Position;
import com.bxm.adx.common.utils.AdxUtils;
import com.bxm.mcssp.common.entity.PositionJsonConfig;
import com.bxm.mcssp.common.enums.position.PositionSceneTypeEnum;
import com.bxm.warcar.cache.Counter;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.utils.DateHelper;
import com.bxm.warcar.utils.KeyBuilder;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import java.util.List;
import java.util.Objects;

/**
 * <h3>推送广告位频次限制过滤器</h3>
 *
 * @author hcmony
 * @since V1.0.0, 2020/11/23 10:46
 */
@Slf4j
@Component
public class PositionPushTimesCounterFilter extends AbstractPositionFilter {

    @Autowired
    private Counter counter;

    @Override
    protected List<String> doFilter(MarketOrders marketOrders) {
        if (marketOrders.getBidRequest().isAppDrivingRequest()) {
            return null;
        }

        final List<Position> positions = marketOrders.getPositions();
        if (CollectionUtils.isEmpty(positions)) {
            return null;
        }

        List<String> list = Lists.newArrayList();

        //需要注意的是这里的position信息已经全部传过来了
        positions.forEach(position -> {
            Byte pScene = position.getPositionScene();
            if (pScene == null) {
                return;
            }

            PositionSceneTypeEnum typeEnum = PositionSceneTypeEnum.get(pScene.intValue());
            if (typeEnum == null) {
                return;
            }
            Integer frequency = null;
            String jsonConfig = position.getJsonConfig();
            switch (typeEnum) {
                case PUSH:
                    if (org.apache.commons.lang3.StringUtils.isNotEmpty(jsonConfig)) {
                        PositionJsonConfig positionJsonConfig = AdxUtils.getJsonConfig(jsonConfig);
                        if (Objects.nonNull(positionJsonConfig)) {
                            if (Objects.nonNull(positionJsonConfig.getPush())) {
                                if (Objects.nonNull(positionJsonConfig.getPush().getPushMaxShowPerDay())) {
                                    frequency = positionJsonConfig.getPush().getPushMaxShowPerDay();
                                    break;
                                }
                            }
                        }
                    }
                    frequency = position.getPushTimesPerPersonPerDay();
                    break;
                case LOCK_SCREEN:
                case SCREEN:
                    if (StringUtils.isEmpty(jsonConfig)) {
                        return;
                    }
                    PositionJsonConfig positionJsonConfig = AdxUtils.getJsonConfig(jsonConfig);
                    if (positionJsonConfig == null) {
                        return;
                    }
                    if (typeEnum == PositionSceneTypeEnum.LOCK_SCREEN) {
                        PositionJsonConfig.LockScreen lockScreen = positionJsonConfig.getLockScreen();
                        if (lockScreen == null) {
                            return;
                        }
                        if (lockScreen.getShowTimesPerPersonPerDay() == null) {
                            return;
                        }
                        frequency = lockScreen.getShowTimesPerPersonPerDay();
                    }
                    if (typeEnum == PositionSceneTypeEnum.SCREEN) {
                        PositionJsonConfig.Screen screen = positionJsonConfig.getScreen();
                        if (screen == null) {
                            return;
                        }
                        if (screen.getExposurePerPersonPerDay() == null) {
                            return;
                        }
                        frequency = screen.getExposurePerPersonPerDay();
                    }
                    break;
                default:
                    return;
            }

            if (frequency == null) {
                return;
            }
            //数据来自于adxcounter
            KeyGenerator key = () -> KeyBuilder.build("statistics", "adx", "show", "user", DateHelper.getDate(), position.getPositionId());
            final Long count = counter.hget(key, marketOrders.getUid());

            if (count == null || count < frequency) {
                return;
            }

            list.add(position.getPositionId());
        });

        return list;
    }

    @Override
    public int order() {
        return 1;
    }
}
