package com.bxm.adx.common.buy.buyers;

import com.bxm.adx.common.AdxProperties;
import com.bxm.adx.common.adapter.AdxContextFactory;
import com.bxm.adx.common.buy.Buyer;
import com.bxm.adx.common.buy.dispatcher.Dispatcher;
import com.bxm.adx.common.buy.dispatcher.DispatcherService;
import com.bxm.adx.common.buy.dsp.Dsp;
import com.bxm.adx.common.buy.dsp.DspService;
import com.bxm.adx.common.buy.optimization.Optimization;
import com.bxm.adx.common.buy.optimization.OptimizationDao;
import com.bxm.adx.common.ip.IpService;
import com.bxm.adx.common.plugin.PluginHolder;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.position.Position;
import com.bxm.adx.common.sell.request.Device;
import com.bxm.adx.facade.exception.AdxException;
import com.bxm.warcar.ip.IP;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;

import java.math.BigDecimal;
import java.util.*;

/**
 * @author allen
 * @since 2019-12-16
 */
@Primary
@Configuration
public class DispatcherPluginBuyersImpl extends AbstractPluginBuyersImpl {

    private DispatcherService dispatcherService;
    private final AdxProperties properties;
    private final DspService dspService;
    private final OptimizationDao optimizationDao;
    private final IpService ipService;

    protected DispatcherPluginBuyersImpl(PluginHolder pluginHolder, AdxProperties properties, DspService dspService,
                                         OptimizationDao optimizationDao, IpService ipService) {
        super(pluginHolder);
        this.properties = properties;
        this.dspService = dspService;
        this.optimizationDao = optimizationDao;
        this.ipService = ipService;
    }

    @Override
    public List<List<BuyerWrapper>> findAsPriority(Position position, BigDecimal minPrice, BigDecimal maxPrice) {
        if (Objects.nonNull(minPrice) && Objects.nonNull(maxPrice) && (maxPrice.compareTo(BigDecimal.ZERO) > 0)) {
            AdxContextFactory.get().setSDKConcurrentModel(true);
        } else {
            AdxContextFactory.get().setSDKConcurrentModel(false);
        }

        Map<Integer, Collection<Dispatcher>> dsps = dispatcherService.getPriority(position, minPrice, maxPrice);

        //此处因为系统暂时只支持5级优先级分配，所以设置size为5就够了，实际如果想兼容不限级数的优先级的话需要考虑list的容量动态扩容的问题，
        //list作为接口是没有这个方法的，arraylist有，但是此处list对象不能直接调用arraylist类的方法
        //改于20210426,如果当前最大优先级超过5，则初始化list长度为最大优先级（动态扩容是一种解法，按最大优先级设置是另一种）
        Integer initSize = properties.getMaxCapacityOfBuyers();
        if (!MapUtils.isEmpty(dsps)) {
            Set<Integer> keys = dsps.keySet();
            if (!CollectionUtils.isEmpty(keys)) {
                Integer maxPriority = keys.stream().max(Integer::compareTo).get();
                if (maxPriority != null && maxPriority.compareTo(initSize) == 1) {
                    initSize = maxPriority;
                }
            }
        }

        List<List<BuyerWrapper>> list = emptyListWithCapacity(initSize);

        Map<String, Buyer> buyers = getAllBuyers();
        for (Map.Entry<Integer, Collection<Dispatcher>> entry : dsps.entrySet()) {
            // 优先级，有可能从2开始
            Integer priority = entry.getKey();
            Collection<Dispatcher> value = entry.getValue();

            int index = priority - 1;
            List<BuyerWrapper> buyerList = CollectionUtils.isEmpty(list) ? null : list.size() > index ? list.get(index) : null;

            // 如果当前优先级队列不存在，则构建一个队列
            if (null == buyerList) {
                buyerList = Lists.newArrayList();
                //此处如果写死list容量为5，由于index的值是根据优先级减1得来，当分配dsp进行重排序后，最大优先级会超过5，就会发生数组越界
                list.add(index, buyerList);
            }

            // 将Buyer实例添加到队列中
            for (Dispatcher dispatcher : value) {
                Long dspId = dispatcher.getDspId();
                Dsp dsp = dspService.get(dspId);
                if (null == dsp) {
                    continue;
                }
                String dspCode = dsp.getDspCode();
                Buyer buyer = buyers.get(dspCode);
                if (null != buyer) {
                    buyerList.add(new BuyerWrapper(buyer, dispatcher));
                }
            }
        }

        // 删除没有空的队列
        list.removeIf(List::isEmpty);
        // 如果sdk排序配置中是并行模式（是否是并行模式根据是否传了价格范围来判断，旧版不传价格范围）
        if (AdxContextFactory.get().isSDKConcurrentModel()) {
            List<BuyerWrapper> buyerWrapperList = new ArrayList<>();
            for (List<BuyerWrapper> buyerWrappers : list) {
                if (CollectionUtils.isNotEmpty(buyerWrappers)) {
                    buyerWrapperList.addAll(buyerWrappers);
                }
            }
            List<List<BuyerWrapper>> result = new ArrayList<>();
            result.add(buyerWrapperList);
            return result;
        }

        return list;
    }

    @Override
    public void rebuildBuyers(BidRequest bidRequest, Position position, List<List<BuyerWrapper>> list) {
        handleBackupEntrance(bidRequest, list);
        handleOptimization(position, bidRequest, list);
    }

    private List<List<BuyerWrapper>> emptyListWithCapacity(int size) {
        List<List<BuyerWrapper>> list = Lists.newArrayList();
        for (int i = 0; i < size; i++) {
            list.add(Lists.newArrayList());
        }
        return list;
    }

    @Autowired
    public void setDispatcherService(DispatcherService dispatcherService) {
        this.dispatcherService = dispatcherService;
    }

    /**
     * 处理兜底请求
     * @param request
     * @param buyers
     */
    private void handleBackupEntrance(BidRequest request, List<List<BuyerWrapper>> buyers) {
        String entranceName = request.getEntrance_name();
        if (StringUtils.isEmpty(entranceName)) {
            return;
        }
        if ("backup".equalsIgnoreCase(entranceName)) {
            buyers.clear();
            List<BuyerWrapper> buyerList = new ArrayList<>();
            buyers.add(buyerList);
            Map<String, Buyer> allBuyers = getAllBuyers();
            Buyer buyer = allBuyers.get("pangu-backup");
            if (buyer == null) {
                throw new AdxException("can't find backup-dsp ");
            }
            Dispatcher dispatcher = new Dispatcher();
            dispatcher.setDspBasePrice(BigDecimal.ZERO);
            dispatcher.setDspId(buyer.getDsp().getId());
            dispatcher.setDspAvgPrice(BigDecimal.ZERO);
            buyerList.add(new BuyerWrapper(buyer, dispatcher));
        }
    }


    /**
     * 处理开启广告优化的广告位: 修改流量分配，添加流量优化DSP
     * @param position
     * @param request
     * @param buyers
     */
    private void handleOptimization(Position position, BidRequest request, List<List<BuyerWrapper>> buyers) {
        Optimization optimization = optimizationDao.getOptimizationByPositionId(position.getPositionId());
        if (Objects.isNull(optimization)) {
            return;
        }
        String regionCodes = optimization.getRegion();
        if (!org.springframework.util.StringUtils.isEmpty(regionCodes)) {
            IP ipp = getIp(request);
            Set<String> set = Sets.newHashSet();
            set.addAll(Arrays.asList(regionCodes.split(",")));
            if (Objects.isNull(ipp) || !ipp.in(set)) {
                return;
            }
        }

        Map<String, Buyer> allBuyers = getAllBuyers();
        Buyer buyer = allBuyers.get("optimization");
        if (buyer == null) {
            throw new AdxException("can't find op-dsp ");
        }
        Dispatcher dispatcher = new Dispatcher();
        dispatcher.setDspPosid(optimization.getDspPositionId());
        buyers.add(Lists.newArrayList(new BuyerWrapper(buyer, dispatcher)));
        AdxContextFactory.get().setOptimization(true);
    }

    private IP getIp(BidRequest bidRequest) {
        IP ipp = null;
        if (bidRequest != null) {
            Device device = bidRequest.getDevice();
            if (device != null) {
                String ip = device.getIp();
                if (!org.springframework.util.StringUtils.isEmpty(ip)) {
                    ipp = ipService.analyze(ip);
                    AdxContextFactory.get().setIp(ipp);
                }
            }
        }
        return ipp;
    }
}
