package com.bxm.adx.common.adapter;

import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.adapter.builder.Interpreter;
import com.bxm.adx.common.cipher.CipherService;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.BidResponse;
import com.bxm.adx.common.sell.ssp.SspRequest;
import com.bxm.adx.common.sell.ssp.SspResponse;
import com.bxm.adx.common.sell.ssp.resp.Ad;
import com.bxm.adx.common.utils.AdxUtils;
import com.bxm.adx.facade.constant.enums.AdxErrEnum;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.StringHelper;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.digest.DigestUtils;

import javax.annotation.Resource;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Objects;

/**
 * Created by Arei on 2020/1/16.
 */
@Slf4j
public class BxmSspBidModelAdapter implements BidModelAdapter {

    @Resource
    private Interpreter interpreter;
    @Resource
    private CipherService cipherService;
    @Resource
    private ObjectMapper objectMapper;

    @Override
    public BidRequest convert(byte[] bytes) {
        SspRequest sspRequest = JsonHelper.convert(bytes, SspRequest.class);
        BidRequest request = new BidRequest();
        interpreter.translate(sspRequest, request);
        if (log.isDebugEnabled()) {
            log.debug("bidrequest={}", JsonHelper.convert(request));
        }

        AdxContext adxContext = AdxContextFactory.get();
        adxContext.setAttachment(sspRequest);
//        adxContext.setUid(AdxUtils.getUid(request));
        adxContext.setUid(AdxUtils.getUidV2(request));

        return request;
    }

    @Override
    public byte[] convert(BidResponse e, BidRequest bidRequest) {
        if (log.isDebugEnabled()) {
            log.debug("bidresponse={}", JsonHelper.convert(e));
        }
        SspResponse sspResponse = new SspResponse();
        interpreter.translate(e, bidRequest, sspResponse);

        byte[] bytes = JsonHelper.convert2bytes(sspResponse);

        // cipher
        AdxContext context = AdxContextFactory.get();
        Object attachment = context.getAttachment();
        if (attachment instanceof SspRequest && (context.isCipherAesForResponse())) {
            List<Ad> ads = sspResponse.getAds();
            String content = JsonHelper.convert(ads);

            try {
                String key = DigestUtils.md5Hex(JsonHelper.convert2bytes(attachment));
                String ciphertext = cipherService.aesEncrypt(content, key);

                String requestId = cipherService.generateRequestId(key);

                sspResponse.setCiphertext(ciphertext);
                sspResponse.setRequestId(requestId);

                return JsonHelper.convert2bytes(sspResponse);

            } catch (Exception ex) {
                log.warn("encrypt: ", ex);
            }
        }

        return bytes;
    }

    @Override
    public int statusCode() {
        return BidModelAdapter.super.statusCode();
    }

    @Override
    public byte[] reject(BidRequest request, AdxErrEnum adxErrEnum) {
        if (StringHelper.isNewVersion(AdxConstants.SDK_ERR_CODE_VERSION, request.getSdk_ver())) {
            ObjectNode objectNode = objectMapper.createObjectNode();

            if (Objects.nonNull(adxErrEnum)) {
                objectNode.put("result", adxErrEnum.getResult());
                objectNode.put("msg", adxErrEnum.getMsg());
            }

            return objectNode.asText().getBytes(StandardCharsets.UTF_8);
        }
        return BidModelAdapter.super.reject(request, adxErrEnum);
    }
}
