package com.bxm.adx.common.log.dsplog;


import com.alibaba.fastjson.JSON;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.BidResponse;
import com.bxm.warcar.utils.NamedThreadFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * 记录dsp的请求和响应 分开文件进行记录
 * 多线程写文件 会导致主链路耗时增加 因为并发写文件要加锁 1个线程和100个线程没区别
 *
 * @author zhangdong
 * @date 2022/11/30
 */
@Component
public class DspLogRecord implements DisposableBean {


    private final Logger requestLogger = LoggerFactory.getLogger(DspLogRecord.class.getName() + ".request");

    private final Logger responseLogger = LoggerFactory.getLogger(DspLogRecord.class.getName() + ".response");

    private final ThreadPoolExecutor executor = new ThreadPoolExecutor(10, 20, 0, TimeUnit.MILLISECONDS,
            new LinkedBlockingQueue<>(1000), new NamedThreadFactory("dspLogResponse"), new ThreadPoolExecutor.CallerRunsPolicy());

    private final ConcurrentHashMap<String, Long> map = new ConcurrentHashMap<>();

    public void dspLog(BidRequest request, BidResponse response) {
        logRequest(request);
        logResponse(response);
    }

    private void logResponse(BidResponse obj) {
        if (obj == null) {
            return;
        }
        executor.execute(() -> responseLogger.info(JSON.toJSONString(obj)));
    }


    private void logRequest(BidRequest obj) {
        if (obj == null) {
            return;
        }
        if (map.containsKey(obj.getId())) {
            return;
        }
        map.put(obj.getId(), System.currentTimeMillis());
        executor.execute(() -> requestLogger.info(JSON.toJSONString(obj)));
    }

    @Scheduled(cron = "*/1 * * * * ?")
    public void deleteExpireKey() {
        long l = System.currentTimeMillis();
        for (Map.Entry<String, Long> entry : map.entrySet()) {
            if ((l - entry.getValue()) > 1000) {
                map.remove(entry.getKey());
            }
        }
    }


    @Override
    public void destroy() {
        executor.shutdown();
        try {
            executor.awaitTermination(5, TimeUnit.SECONDS);
        } catch (Exception ignored) {

        }
    }
}
