package com.bxm.adx.common.market.exchange;

import com.alibaba.fastjson.JSONException;
import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.adapter.BuyModelAdapter;
import com.bxm.adx.common.buy.Buyer;
import com.bxm.adx.common.buy.cache.BuyerResponseCache;
import com.bxm.adx.common.buy.dispatcher.Dispatcher;
import com.bxm.adx.common.buy.dsp.Dsp;
import com.bxm.adx.common.buy.position.AdvertPoint;
import com.bxm.adx.common.buy.position.AdvertPointService;
import com.bxm.adx.common.creative.replace.ReplaceCreative;
import com.bxm.adx.common.ingetration.AdxCounterServiceIntegration;
import com.bxm.adx.common.log.dsplog.DspLogRecord;
import com.bxm.adx.common.log.datalog.DataLogDao;
import com.bxm.adx.common.market.Deal;
import com.bxm.adx.common.micrometer.BuyerMeter;
import com.bxm.adx.common.openlog.event.internal.AdxRequestEvent;
import com.bxm.adx.common.openlog.event.internal.DspBidEvent;
import com.bxm.adx.common.sell.BidConfig;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.BidResponse;
import com.bxm.adx.common.sell.request.*;
import com.bxm.adx.common.sell.response.Bid;
import com.bxm.adx.common.sell.response.SeatBid;
import com.bxm.adx.common.sell.response.Video;
import com.bxm.adx.facade.constant.enums.AdxErrEnum;
import com.bxm.adx.facade.exception.AdxException;
import com.bxm.adxcounter.facade.model.AdxCounterDTO;
import com.bxm.mccms.facade.enums.OfferSettle;
import com.bxm.mcssp.common.enums.app.DockingMethodTypeEnum;
import com.bxm.warcar.integration.eventbus.EventPark;
import com.bxm.warcar.utils.JsonHelper;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.RandomUtils;
import org.springframework.beans.BeanUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.Callable;
import java.util.concurrent.CountDownLatch;
import java.util.stream.Collectors;

/**
 * @author allen
 * @date 2020-10-30
 * @since 1.0
 */
@Slf4j
public class ExchangeCallable implements Callable<Deal> {

    private final Buyer buyer;
    private final BidRequest bidRequest;
    private final BuyerMeter buyerMeter;
    private final ExchangeParam exchangeParam;
    private final EventPark eventPark;
    private final BuyerResponseCache buyerResponseCache;
    private final CountDownLatch waitCountDown;
    private final CountDownLatch overtimeCountDown;
    private final AdvertPointService advertPointService;
    private final BidConfig bidConfig;
    private final DataLogDao dataLogDao;
    private final DspLogRecord dspLogRecord;

    public ExchangeCallable(Buyer buyer, BidRequest bidRequest,
                            BuyerMeter buyerMeter, ExchangeParam exchangeParam, EventPark eventPark,
                            BuyerResponseCache buyerResponseCache,
                            CountDownLatch waitCountDown, CountDownLatch overtimeCountDown,
                            AdvertPointService advertPointService, BidConfig bidConfig, DataLogDao dataLogDao,
                            DspLogRecord dspLogRecord) {

        this.buyer = buyer;
        this.bidRequest = bidRequest;
        this.buyerMeter = buyerMeter;
        this.exchangeParam = exchangeParam;
        this.eventPark = eventPark;
        this.buyerResponseCache = buyerResponseCache;
        this.waitCountDown = waitCountDown;
        this.overtimeCountDown = overtimeCountDown;
        this.advertPointService = advertPointService;
        this.bidConfig = bidConfig;
        this.dataLogDao = dataLogDao;
        this.dspLogRecord = dspLogRecord;
    }

    private String getNet(Device device) {
        Integer connectionType = device.getConnection_type();
        return Objects.toString(connectionType);
    }

    @Override
    public Deal call() {
        long startTime = System.currentTimeMillis();

        String name = buyer.getCode();
        BuyModelAdapter modelAdapter = buyer.getModelAdapter();
        if (null == modelAdapter) {
            log.warn("BuyModelAdapter [{}] not found!", name);
            return null;
        }

        byte[] request = null;

        BidRequest adxRequest = null;
        Deal deal = null;
        try {
            // 对上下文设置必要的对象
            ExchangeContext.putBidRequest(bidRequest);
            ExchangeContext.putBuyer(buyer);
            ExchangeContext.putDispatchConfig(exchangeParam.getDispatcherConfigId());
            Dispatcher dispatcher = exchangeParam.getDispatcher();
            ExchangeContext.putDispatch(dispatcher);

            adxRequest = rebuildBidRequest(exchangeParam);
            ExchangeContext.putAdxRequest(adxRequest);
            request = modelAdapter.buildRequest(adxRequest);
            if (ArrayUtils.isEmpty(request)) {
                return null;
            }
            Dsp dsp = buyer.getDsp();
            String configId = StringUtils.EMPTY;
            if (Objects.nonNull(dispatcher) && Objects.nonNull(dispatcher.getConfigId())) {
                configId = dispatcher.getConfigId().toString();
            }
            //埋点adx请求
            eventPark.post(new AdxRequestEvent(this, bidRequest, adxRequest, dsp.getId().toString(), configId,
                    bidConfig));

            // 请求指标
            buyerMeter.increaseRequest(buyer);

            byte[] response = offer(request);
            if (ArrayUtils.isEmpty(response)) {
                throw new AdxException(AdxErrEnum.DSP_EMPTY_RESPONSE);
            }

            deal = new Deal(buyer, request, response, bidRequest, adxRequest);

            if (!deal.isBidSuccess()) {
                throw new AdxException(AdxErrEnum.DSP_EMPTY_RESPONSE);
            }
            //记录有广告返回的日志
            dspLogRecord.dspLog(bidRequest, deal.getBidResponse());
            //填充response
            rebuildResponse(deal, exchangeParam, dsp);

            //埋点dsp返回
            eventPark.post(new DspBidEvent(this, bidRequest, adxRequest, deal.getBidResponse(),
                    dsp.getId().toString(), bidConfig));

            // 填充指标
            buyerMeter.increasePadding(buyer);

            return deal;
        } catch (AdxException e) {
            if (log.isDebugEnabled()) {
                log.debug("exchange err: {}", e.getAdxErrEnum().name());
            }
            AdxErrEnum adxErrEnum = Optional.ofNullable(e.getAdxErrEnum()).orElse(AdxErrEnum.UNKNOWN_ERR);
            return new Deal(buyer, request, bidRequest, adxRequest, adxErrEnum);
        } catch (JSONException e) {
            return new Deal(buyer, request, bidRequest, adxRequest, AdxErrEnum.UNKNOWN_ERR);
        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("deal call err", e);
            }
            return new Deal(buyer, request, bidRequest, adxRequest, AdxErrEnum.UNKNOWN_ERR);
        } finally {
            long endTime = System.currentTimeMillis();
            long interval = endTime - startTime;
            if (log.isDebugEnabled()) {
                log.debug("buy = {}, interval = {}", JsonHelper.convert(buyer), interval);
            }
            if (interval < bidRequest.getWaitTime()) {
                waitCountDown.countDown();
            } else if (interval < bidRequest.getOvertime()) {
                overtimeCountDown.countDown();
            } else {
                if (Objects.nonNull(deal) && deal.isBidSuccess()) {
                    buyerResponseCache.saveResponse(deal.getAdxRequest(), deal.getBidResponse(), buyer);
                }
            }
            ExchangeContext.remove();
        }
    }

    private byte[] offer(byte[] request) {
        long start = System.nanoTime();
        byte[] response;
        try {
            response = buyer.offer(request);
        } finally {
            // 耗时指标
            buyerMeter.recordRequest(buyer, start);
        }
        return response;
    }

    /**
     * 重构bidrequest
     * 1.判断广告位接入类型SDK/API决定是否使用流量分配中的广告位id
     * 2.流量方/流量分配两者的底价选择
     * 3.流量方/流量分配两者的APPID选择
     *
     * @param exchangeParam
     * @return
     */
    private BidRequest rebuildBidRequest(ExchangeParam exchangeParam) {
        BidRequest adxRequest = new BidRequest();
        BeanUtils.copyProperties(bidRequest, adxRequest);
        Dispatcher dispatcher = exchangeParam.getDispatcher();
        List<Impression> impressions = adxRequest.getImps();
        Integer dockingMethodType = adxRequest.getDockingMethodType();
        List<Impression> newImpressions = new ArrayList<>();
        AdvertPoint advertPoint = null;
        for (Impression impression : impressions) {
            if (Objects.isNull(impression.getBid_floor())) {
                impression.setBid_floor(BigDecimal.ZERO);
            }
            Impression newImpression = new Impression();
            BeanUtils.copyProperties(impression, newImpression);
            //替换DSP广告位ID
            if (Objects.nonNull(dockingMethodType) &&
                    (dockingMethodType == DockingMethodTypeEnum.SDK_OPERATION.getType())) {
                if (buyer.getCode().equalsIgnoreCase("scene")) {
                    //场景dsp直接用SSP入口广告位
                } else {
                    newImpression.setTag_id(dispatcher.getDspPosid());
                }
            } else {
                if (Objects.isNull(advertPoint)) {
                    advertPoint = advertPointService.getAdvertPointByDspId(dispatcher.getDspId().toString(), dispatcher.getDspAppid(), dispatcher.getDspPosid());
                }
                if (Objects.nonNull(advertPoint)) {
                    AdvertPoint.Size size = advertPoint.getImpressionSize();
                    if (Objects.nonNull(size)) {
                        newImpression.setH(size.getH());
                        newImpression.setW(size.getW());
                    }
                    if (StringUtils.isNotEmpty(advertPoint.getTemplateId())) {
                        Native an = new Native();
                        if (Objects.nonNull(newImpression.getA_native())) {
                            BeanUtils.copyProperties(newImpression.getA_native(), an);
                        }
                        an.setTemplate(advertPoint.getTemplateId());
                        newImpression.setA_native(an);
                    }
                }
                ExchangeContext.putDataLog(dataLogDao.existDataLog(impression.getTag_id(), dispatcher.getDspPosid()));
                newImpression.setTag_id(dispatcher.getDspPosid());
            }
            //优先使用流量分配中的底价
            BigDecimal dspBasePrice = dispatcher.getDspBasePrice();
            if (Objects.nonNull(dspBasePrice) && dspBasePrice.movePointRight(2).compareTo(impression.getBid_floor()) > 0) {
                newImpression.setBid_floor(dispatcher.getDspBasePrice().movePointRight(2));
            }
            if (Objects.isNull(newImpression.getBid_floor())) {
                newImpression.setBid_floor(BigDecimal.ZERO);
            }
            //如果利润率不为空， （新底价 = 原底价/（100 - 利润率））
            if (Objects.nonNull(dispatcher.getProfitMargin())) {
                BigDecimal floor = newImpression.getBid_floor().divide(new BigDecimal(100).subtract(dispatcher.getProfitMargin()).movePointLeft(2), 2, BigDecimal.ROUND_UP);
                newImpression.setBid_floor(floor);

            }

            //底价单位为分，不保留小数，有小数自动向上进位
            newImpression.setBid_floor(newImpression.getBid_floor().setScale(0, BigDecimal.ROUND_UP));
            newImpressions.add(newImpression);
            break;
        }
        adxRequest.setImps(newImpressions);

        App app = new App();
        if (Objects.nonNull(adxRequest.getApp())) {
            BeanUtils.copyProperties(adxRequest.getApp(), app);
        }
        app.setId(dispatcher.getDspAppid());
        if (StringUtils.isNotEmpty(dispatcher.getAppPackageName())) {
            app.setBundle(dispatcher.getAppPackageName());
        }
        if (Objects.nonNull(advertPoint)) {
            app.setVer(advertPoint.getVersion());
            app.setName(advertPoint.getAppName());
        }

        adxRequest.setApp(app);
        return adxRequest;
    }

    /**
     * 重构Response
     *
     * @param deal
     * @param exchangeParam
     */
    private void rebuildResponse(Deal deal, ExchangeParam exchangeParam, Dsp dsp) {
        if (deal.isOp()) {
            return;
        }

        BidResponse bidResponse = deal.getBidResponse();
        BidRequest bidRequest = deal.getBidRequest();
        Dispatcher dispatcher = exchangeParam.getDispatcher();
        if (StringUtils.isEmpty(bidResponse.getId())) {
            bidResponse.setId(bidRequest.getId());
        }

        for (SeatBid seatBid : bidResponse.getSeat_bid()) {
            seatBid.setDspId(dispatcher.getDspId());
            seatBid.setConfigId(dispatcher.getConfigId());
            for (Bid bid : seatBid.getBid()) {
                bid.setCharge_type(AdxConstants.ChargeType.CPM);
                //记录dsp出价
                bid.setDsp_price(new BigDecimal(bid.getPrice().toString()));
                if (Objects.nonNull(dispatcher.getDspAvgPrice())) {
                    String mediaSettle = dispatcher.getMediaSettle();
                    if (Objects.nonNull(dispatcher.getMediaSettle())) {
                        if (OfferSettle.CPC.name().equalsIgnoreCase(mediaSettle)) {
                            bid.setCharge_type(AdxConstants.ChargeType.CPC);
                            bid.setBid(dispatcher.getDspAvgPrice().movePointRight(2));
                        } else {
                            bid.setPrice(dispatcher.getDspAvgPrice().movePointRight(2));
                        }
                    } else {
                        bid.setPrice(dispatcher.getDspAvgPrice().movePointRight(2));
                    }
                } else {
                    bid.setPrice(bid.getPrice());
                }
                //如果利润率不为空，则计算向流量方的出价替换原有DSP的广告出价 （新出价 = （100 - 利润率）* DSP出价）
                if (Objects.nonNull(dispatcher.getProfitMargin())) {
                    BigDecimal price = bid.getPrice();
                    price = price.multiply(new BigDecimal(100).subtract(dispatcher.getProfitMargin())).movePointLeft(2);
                    bid.setPrice(price);
                }
                //出价舍去小数（向流量方出价更低，提高收益）
                bid.setPrice(bid.getPrice().setScale(0, BigDecimal.ROUND_DOWN));
                bid.setDsp_appid(exchangeParam.getDispatcher().getDspAppid());
                bid.setImp_id(bidRequest.getImps().iterator().next().getId());

                ReplaceCreative creative = randomCreative(dispatcher.getCreativeList());
                com.bxm.adx.common.sell.response.Native an = bid.getA_native();
                //暂不支持如果预算完全没有素材返回的情况下替换素材
                if (Objects.nonNull(creative) && Objects.nonNull(an) && CollectionUtils.isNotEmpty(an.getAssets())) {
                    buildNative(an, creative);
                    bid.setAdxCreateId(creative.getId().toString());
                }
            }
        }
    }

    /**
     * 随机出替换素材
     *
     * @param creatives
     * @return
     */
    private ReplaceCreative randomCreative(List<ReplaceCreative> creatives) {
        if (CollectionUtils.isNotEmpty(creatives)) {
            int r = RandomUtils.nextInt(creatives.size());
            return creatives.get(r);
        }
        return null;
    }

    /**
     * 替换DSP的素材
     *
     * @param an
     * @param creative
     * @return
     */
    private void buildNative(com.bxm.adx.common.sell.response.Native an, ReplaceCreative creative) {
        List<com.bxm.adx.common.sell.response.Asset> needReplace = needReplace(creative);
        if (CollectionUtils.isEmpty(needReplace)) {
            return;
        }
        Set<Integer> replacedTypes = Sets.newHashSet();
        List<com.bxm.adx.common.sell.response.Asset> newAsset = new ArrayList<>();
        for (com.bxm.adx.common.sell.response.Asset asset : an.getAssets()) {
            Integer type = asset.getType();
            if (Objects.isNull(type)) {
                continue;
            }
            Optional<com.bxm.adx.common.sell.response.Asset> replace = null;
            if (type.equals(AdxConstants.AssetType.REWARDER_VIDEO.getType())) {
                replace = needReplace.stream()
                        .filter(a -> a.getType().equals(AdxConstants.AssetType.VIDEO.getType()))
                        .findFirst();
            } else {
                replace = needReplace.stream()
                        .filter(a -> a.getType().equals(type))
                        .findFirst();
            }
            if (replace.isPresent()) {
                com.bxm.adx.common.sell.response.Asset replaceAsset = replace.get();
                replacedTypes.add(replaceAsset.getType());
                if (type.equals(AdxConstants.AssetType.VIDEO.getType()) || type.equals(AdxConstants.AssetType.REWARDER_VIDEO.getType())) {
                    Video oldVideo = asset.getVideo();
                    Video newVideo = replaceAsset.getVideo();
                    if (Objects.nonNull(oldVideo)) {
                        newVideo.setV_monitor(oldVideo.getV_monitor());
                        if (Objects.nonNull(newVideo.getCover_url())) {
                            oldVideo.setCover_url(newVideo.getCover_url());
                        }
                        if (Objects.nonNull(newVideo.getW())) {
                            oldVideo.setW(newVideo.getW());
                        }
                        if (Objects.nonNull(newVideo.getH())) {
                            oldVideo.setH(newVideo.getH());
                        }
                        if (Objects.nonNull(newVideo.getDuration())) {
                            oldVideo.setDuration(newVideo.getDuration());
                        }
                        if (Objects.nonNull(newVideo.getMax_length())) {
                            oldVideo.setMax_length(newVideo.getMax_length());
                        }
                        if (Objects.nonNull(newVideo.getUrl())) {
                            oldVideo.setUrl(newVideo.getUrl());
                        }
                        if (Objects.nonNull(newVideo.getMime_types())) {
                            oldVideo.setMime_types(newVideo.getMime_types());
                        }
                    }
                    com.bxm.adx.common.sell.response.Asset newVideoAsset = com.bxm.adx.common.sell.response.Asset.builder()
                            .type(type)
                            .video(oldVideo)
                            .build();
                    newAsset.add(newVideoAsset);
                    continue;
                }
                // FIXME: 2022/11/24 FastJson对重复/循环引用的处理
                com.bxm.adx.common.sell.response.Asset copyAsset = JsonHelper.convert(JsonHelper.convert(replaceAsset), com.bxm.adx.common.sell.response.Asset.class);
                newAsset.add(copyAsset);
            } else {
                newAsset.add(asset);
            }
        }
        List<com.bxm.adx.common.sell.response.Asset> needAdd = needReplace.stream()
                .filter(asset -> !replacedTypes.contains(asset.getType()))
                .collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(needAdd)) {
            newAsset.addAll(needAdd);
        }
        an.setAssets(newAsset);
    }

    private List<com.bxm.adx.common.sell.response.Asset> needReplace(ReplaceCreative creative) {
        List<com.bxm.adx.common.sell.response.Asset> assets = new ArrayList<>();
        Integer type = creative.getType();

        String title = creative.getTitle();
        String content = creative.getContent();
        String img = creative.getImageUrl();
        String icon = creative.getIconUrl();

        if (StringUtils.isNotEmpty(title)) {
            com.bxm.adx.common.sell.response.Asset titleAsset =
                    com.bxm.adx.common.sell.response.Asset.builder()
                            .type(AdxConstants.AssetType.TITLE.getType())
                            .text(com.bxm.adx.common.sell.response.Text.builder()
                                    .text(title)
                                    .build())
                            .build();
            assets.add(titleAsset);
        }
        if (StringUtils.isNotEmpty(content)) {
            com.bxm.adx.common.sell.response.Asset contentAsset =
                    com.bxm.adx.common.sell.response.Asset.builder()
                            .type(AdxConstants.AssetType.CONTENT.getType())
                            .text(com.bxm.adx.common.sell.response.Text.builder()
                                    .text(content)
                                    .build())
                            .build();
            assets.add(contentAsset);
        }

        if (StringUtils.isNotEmpty(img)) {
            String imageSize = creative.getImageSize();
            String[] sizes = null;
            Integer w = null;
            Integer h = null;
            if (StringUtils.isNotEmpty(imageSize)) {
                try {
                    sizes = imageSize.split("\\*");
                    w = Integer.valueOf(sizes[0]);
                    h = Integer.valueOf(sizes[1]);
                } catch (Exception e) {
                }
            }
            if (type.equals(ReplaceCreative.TYPE_IMG)) {
                com.bxm.adx.common.sell.response.Asset imgAsset =
                        com.bxm.adx.common.sell.response.Asset.builder()
                                .type(AdxConstants.AssetType.LARGE_IMG.getType())
                                .img(com.bxm.adx.common.sell.response.Image.builder()
                                        .h(h)
                                        .w(w)
                                        .url(img)
                                        .build())
                                .build();
                assets.add(imgAsset);
            }
        }
        if (StringUtils.isNotEmpty(icon)) {
            String iconSize = creative.getIconSize();
            String[] sizes = null;
            Integer w = null;
            Integer h = null;
            if (StringUtils.isNotEmpty(iconSize)) {
                try {
                    sizes = iconSize.split("\\*");
                    w = Integer.valueOf(sizes[0]);
                    h = Integer.valueOf(sizes[1]);
                } catch (Exception e) {
                }
            }
            com.bxm.adx.common.sell.response.Asset iconAsset =
                    com.bxm.adx.common.sell.response.Asset.builder()
                            .type(AdxConstants.AssetType.ICON.getType())
                            .img(com.bxm.adx.common.sell.response.Image.builder()
                                    .h(h)
                                    .w(w)
                                    .url(icon)
                                    .build())
                            .build();
            assets.add(iconAsset);
        }

        if (type.equals(ReplaceCreative.TYPE_VIDEO)) {

            String videoSize = creative.getVideoSize();
            String[] sizes = null;
            Integer w = null;
            Integer h = null;
            if (StringUtils.isNotEmpty(videoSize)) {
                try {
                    sizes = videoSize.split("\\*");
                    w = Integer.valueOf(sizes[0]);
                    h = Integer.valueOf(sizes[1]);
                } catch (Exception e) {
                }
            }
            Video video = Video.builder()
                    .w(w)
                    .h(h)
                    .duration(Objects.isNull(creative.getVideoTime()) ? null : creative.getVideoTime() * 1000)
                    .cover_url(creative.getImageUrl())
                    .mime_types(creative.getVideoFormat())
                    .url(creative.getVideoUrl())
                    .max_length(Objects.nonNull(creative.getVideoKb()) ? creative.getVideoKb().intValue() : null)
                    .build();
            com.bxm.adx.common.sell.response.Asset videoAsset = com.bxm.adx.common.sell.response.Asset.builder()
                    .video(video)
                    .type(AdxConstants.AssetType.VIDEO.getType())
                    .build();
            assets.add(videoAsset);
        }

        return assets;
    }
}
