package com.bxm.adx.common.market.exchange;

import com.bxm.adx.common.AdxProperties;
import com.bxm.adx.common.DspThreadPoolProperties;
import com.bxm.adx.common.adapter.AdxContextFactory;
import com.bxm.adx.common.buy.buyers.BuyerWrapper;
import com.bxm.adx.common.buy.cache.BuyerResponseCache;
import com.bxm.adx.common.buy.dsp.Dsp;
import com.bxm.adx.common.buy.position.AdvertPointService;
import com.bxm.adx.common.log.datalog.DataLogDao;
import com.bxm.adx.common.log.dsplog.DspLogRecord;
import com.bxm.adx.common.market.Deal;
import com.bxm.adx.common.market.filter.PriceLowerFilter;
import com.bxm.adx.common.micrometer.BuyerMeter;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.warcar.integration.eventbus.EventPark;
import com.bxm.warcar.utils.NamedThreadFactory;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.concurrent.*;

/**
 * @author allen
 * @since 2019-12-12
 */
@Slf4j
@Configuration
public class RtbExchanger implements Exchanger, DisposableBean {

    private final BuyerMeter buyerMeter;
    private final EventPark eventPark;
    private final BuyerResponseCache buyerResponseCache;
    private final PriceLowerFilter priceLowerFilter;
    private final AdvertPointService advertPointService;
    private final DataLogDao dataLogDao;
    private final ThreadPoolExecutor executor;
    private final DspLogRecord dspLogRecord;

    public RtbExchanger(BuyerMeter buyerMeter, EventPark eventPark, BuyerResponseCache buyerResponseCache,
                        PriceLowerFilter priceLowerFilter, AdvertPointService advertPointService, DataLogDao dataLogDao,
                        CustomRejected customRejected, AdxProperties adxProperties, DspLogRecord dspLogRecord) {
        this.buyerMeter = buyerMeter;
        this.eventPark = eventPark;
        this.buyerResponseCache = buyerResponseCache;
        this.priceLowerFilter = priceLowerFilter;
        this.advertPointService = advertPointService;
        this.dataLogDao = dataLogDao;
        this.dspLogRecord = dspLogRecord;
        DspThreadPoolProperties dspThreadPool = adxProperties.getDspThreadPool();
        this.executor = new ThreadPoolExecutor(dspThreadPool.getCoreCount(), dspThreadPool.getMaxCount(), dspThreadPool.getLiveTime(), TimeUnit.MILLISECONDS,
                new LinkedBlockingQueue<>(dspThreadPool.getQueueLength()), new NamedThreadFactory("exchange"), customRejected);
    }

    @Override
    public List<Deal> bidding(BidRequest bidRequest, Collection<BuyerWrapper> buyers) {
        if (0 == buyers.size() || null == bidRequest) {
            return Lists.newArrayList();
        }
        List<Future<Deal>> futures = Lists.newArrayList();
        CountDownLatch waitCountDown = new CountDownLatch(buyers.size());
        CountDownLatch overtimeCountDown = new CountDownLatch(1);
        //遍历提交到线程池
        for (final BuyerWrapper buyer : buyers) {
            //发送http请求
            ExchangeParam exchangeParam = new ExchangeParam();
            exchangeParam.setDispatcher(buyer.getDispatcher());
            futures.add(executor.submit(new ExchangeCallable(buyer.getBuyer(), bidRequest, buyerMeter,
                    exchangeParam, eventPark, buyerResponseCache, waitCountDown, overtimeCountDown, advertPointService,
                    AdxContextFactory.get().getBidConfig(), dataLogDao, dspLogRecord)));
            //查询缓存数据
            Dsp dsp = buyer.getBuyer().getDsp();
            Long cacheTime = dsp.getCacheTime();
            if (Objects.isNull(cacheTime) || cacheTime.intValue() == 0) {
                continue;
            }
            futures.add(executor.submit(new ExchangeCacheCallable(bidRequest, buyer, buyerResponseCache)));
        }
        //正常处理
        List<Deal> deals = waitDealFilter(waitCountDown, bidRequest.getWaitTime(), futures);
        if (!CollectionUtils.isEmpty(deals)) {
            return deals;
        }
        if (futures.size() == 0) {
            return Collections.emptyList();
        }
        //超时处理
        deals = waitDealFilter(overtimeCountDown, bidRequest.getOvertime() - bidRequest.getWaitTime(), futures);
        if (!CollectionUtils.isEmpty(deals)) {
            return deals;
        }
        return Collections.emptyList();
    }

    private List<Deal> waitDealFilter(CountDownLatch waitCountDown, Long waitTime, List<Future<Deal>> dealList) {
        try {
            waitCountDown.await(waitTime, TimeUnit.MILLISECONDS);
        } catch (InterruptedException e) {
            log.error("countDown err", e);
            return null;
        }
        //获取结果
        List<Deal> deals = Lists.newArrayList();
        Iterator<Future<Deal>> iterator = dealList.iterator();
        while (iterator.hasNext()) {
            Future<Deal> future = iterator.next();
            try {
                if (future.isDone()) {
                    Deal response = future.get();
                    if (null != response) {
                        deals.add(response);
                    }
                    iterator.remove();
                }
            } catch (InterruptedException | ExecutionException e) {
                // 超时或意外终止
                future.cancel(true);
                log.error("execute: ", e);
            }
        }
        //价格过滤
        priceLowerFilter.filter(deals);
        return CollectionUtils.isEmpty(deals) ? Lists.newArrayList() : deals;
    }

    @Override
    public void destroy() {
        executor.shutdown();
    }
}
