package com.bxm.adx.common.micrometer.interceptor;

import com.bxm.warcar.utils.SafeMapHelper;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.Timer;
import io.micrometer.core.instrument.binder.MeterBinder;
import org.springframework.web.servlet.HandlerInterceptor;
import org.springframework.web.servlet.ModelAndView;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

/**
 * ADX 请求统计拦截器。
 *
 * @author allen
 * @date 2021-11-17
 * @since 1.0
 */
public class AdxRequestsInterceptor implements MeterBinder, HandlerInterceptor {

    private final ThreadLocal<Long> threadLocal = ThreadLocal.withInitial(new Supplier<Long>() {
        @Override
        public Long get() {
            return System.currentTimeMillis();
        }
    });
    private final ConcurrentHashMap<String, Timer> map = new ConcurrentHashMap<>();
    private MeterRegistry registry;

    @Override
    public void bindTo(MeterRegistry registry) {
        this.registry = registry;
    }

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) throws Exception {
        threadLocal.set(System.nanoTime());
        return true;
    }

    @Override
    public void postHandle(HttpServletRequest request, HttpServletResponse response, Object handler, ModelAndView modelAndView) throws Exception {

    }

    @Override
    public void afterCompletion(HttpServletRequest request, HttpServletResponse response, Object handler, Exception ex) throws Exception {
        try {
            Long start = threadLocal.get();
            String contextPath = request.getRequestURI();
            Timer timer = SafeMapHelper.get(map, contextPath, new SafeMapHelper.InitializingValue<Timer>() {
                @Override
                public Timer initializing() {
                    return Timer.builder("adx.requests").tag("path", contextPath).register(registry);
                }
            });
            if (Objects.nonNull(timer)) {
                timer.record(System.nanoTime() - start, TimeUnit.NANOSECONDS);
            }
        } finally {
            threadLocal.remove();
        }
    }
}
