package com.bxm.adx.common.openlog.listener.internal;

import com.bxm.adx.common.AdxProperties;
import com.bxm.adx.common.DotEventControlProperties;
import com.bxm.adx.common.OpenlogConstants;
import com.bxm.adx.common.openlog.event.internal.DspBidEvent;
import com.bxm.adx.common.sell.builder.BuildAttribute;
import com.bxm.adx.common.sell.builder.BuildAttributeStringMacrosHandler;
import com.bxm.adx.facade.constant.redis.AdxKeyGenerator;
import com.bxm.openlog.extension.client.OpenLogClient;
import com.bxm.warcar.cache.Counter;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.integration.eventbus.EventListener;
import com.bxm.warcar.integration.eventbus.core.AllowConcurrentEvents;
import com.bxm.warcar.integration.eventbus.core.Subscribe;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Configuration;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * @author fgf
 * @date 2022/6/1
 **/
@Configuration
public class DoDspBidEventListener implements EventListener<DspBidEvent> {

    private final int EXPIRE_SECONDS_12_HOURS = 12 * 3600;
    private final int EXPIRE_SECONDS_24_HOURS = 24 * 3600;

    private final Counter counter;
    private OpenLogClient openLogClient;
    private final AdxProperties properties;
    private final BuildAttributeStringMacrosHandler macrosHandler;

    public DoDspBidEventListener(Counter counter, AdxProperties properties, BuildAttributeStringMacrosHandler macrosHandler) {
        this.counter = counter;
        this.properties = properties;
        this.macrosHandler = macrosHandler;
    }

    @Autowired
    public void setOpenLogClient(OpenLogClient openLogClient) {
        this.openLogClient = openLogClient;
    }

    @Override
    @Subscribe
    @AllowConcurrentEvents
    public void consume(DspBidEvent event) {
        handleDspAvgFill(event);
        if (event.getBidConfig().getDisDot()) {
            return;
        }
        String url = properties.getOpenLog().create(true, OpenlogConstants.Mt.DSP_RESPONSE_MT, OpenlogConstants.WITHOUT_MACROS_INNER);
        BuildAttribute attribute = new BuildAttribute(OpenlogConstants.Mt.DSP_RESPONSE_MT)
                .setDspId(event.getDspId())
                .setSspRequest(event.getSspRequest())
                .setAdxRequest(event.getAdxRequest())
                .setDspResponse(event.getDspResponse());
        url = macrosHandler.replaceAll(url, attribute);
        openLogClient.asyncRequest(url);
    }

    /**
     * 通过dsp单位时间内的填充数来控制是否请求dsp
     * @param event
     */
    private void handleDspAvgFill(DspBidEvent event) {
        String dspId = event.getDspId();
        String dspTagId = event.getAdxRequest().getImps().iterator().next().getTag_id();

        LocalDateTime now = LocalDateTime.now();
        DotEventControlProperties control = needControl(dspId, dspTagId, now);
        if (Objects.isNull(control)) {
            return;
        }

        KeyGenerator sectionKey = AdxKeyGenerator.Counter.getDspPositionFillBySection(control.getId(), control.getSectionId(now),
                dspId, dspTagId);

        KeyGenerator controlKey = AdxKeyGenerator.Counter.getDspPositionFillByControl(control.getId(), dspId, dspTagId);

        counter.incrementAndGet(sectionKey, EXPIRE_SECONDS_12_HOURS);
        counter.incrementAndGet(controlKey, EXPIRE_SECONDS_24_HOURS);
    }

    private DotEventControlProperties needControl(String dspId, String dspTagId, LocalDateTime now) {
        List<DotEventControlProperties> controls = properties.getDotEventControls();
        if (CollectionUtils.isEmpty(controls)) {
            return null;
        }

        Optional<DotEventControlProperties> optional = controls.stream()
                .filter(p -> Objects.nonNull(p.getMt()) && OpenlogConstants.Mt.DSP_RESPONSE_MT == p.getMt())
                .filter(p -> p.getDspId().equals(dspId))
                .filter(p -> p.getDspPositionId().equals(dspTagId))
                .filter(p -> now.isAfter(p.getStart()) && now.isBefore(p.getEnd()))
                .findFirst();
        return optional.isPresent() ?  optional.get() : null;
    }
}
