package com.bxm.adx.common.market;

import com.bxm.adx.common.adapter.BuyModelAdapter;
import com.bxm.adx.common.buy.Buyer;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.BidResponse;
import com.bxm.adx.common.sell.response.Bid;
import com.bxm.adx.common.sell.response.SeatBid;
import com.bxm.adx.facade.constant.enums.AdxErrEnum;
import org.apache.commons.lang.RandomStringUtils;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 成交信息，包含买家、卖家等信息
 *
 * @author allen
 * @since 2019-12-13
 */
public class Deal {

    private Buyer buyer;

    private byte[] request;
    private byte[] response;

    private BidRequest bidRequest;
    private BidResponse bidResponse;
    private BidRequest adxRequest;

    private AdxErrEnum adxErrEnum;

    public Deal(Buyer buyer, BidRequest bidRequest, BidResponse bidResponse) {
        this.buyer = buyer;
        this.bidRequest = bidRequest;
        this.bidResponse = bidResponse;
    }

    public Deal(Buyer buyer, byte[] request, BidRequest bidRequest, BidRequest adxRequest, BidResponse bidResponse) {
        this.buyer = buyer;
        this.request = request;
        this.bidRequest = bidRequest;
        this.bidResponse = bidResponse;
        this.adxRequest = adxRequest;
    }

    public Deal(Buyer buyer, byte[] request, byte[] response, BidRequest bidRequest) {
        this.buyer = buyer;
        this.request = request;
        this.response = response;
        this.bidRequest = bidRequest;
    }

    public Deal(Buyer buyer, byte[] request, byte[] response, BidRequest bidRequest, BidResponse bidResponse) {
        this.buyer = buyer;
        this.request = request;
        this.response = response;
        this.bidRequest = bidRequest;
        this.bidResponse = bidResponse;
    }

    public Deal(Buyer buyer, byte[] request, BidRequest bidRequest, AdxErrEnum adxErrEnum) {
        this.buyer = buyer;
        this.request = request;
        this.bidRequest = bidRequest;
        this.adxErrEnum = adxErrEnum;
    }

    public Deal(Buyer buyer, byte[] request, byte[] response, BidRequest bidRequest, BidRequest adxRequest) {
        this.buyer = buyer;
        this.request = request;
        this.response = response;
        this.bidRequest = bidRequest;
        this.adxRequest = adxRequest;
    }

    public Deal(Buyer buyer, byte[] request, BidRequest bidRequest, BidRequest adxRequest, AdxErrEnum adxErrEnum) {
        this.buyer = buyer;
        this.request = request;
        this.bidRequest = bidRequest;
        this.adxRequest = adxRequest;
        this.adxErrEnum = adxErrEnum;
    }

    /**
     * @return 是否竞价成功，有广告返回
     */
    public boolean isBidSuccess() {
        return Objects.nonNull(getBidResponse());
    }

    public BidResponse getBidResponse() {
        if (null == bidResponse) {
            BuyModelAdapter modelAdapter = buyer.getModelAdapter();
            if (null != response && response.length > 0) {
                this.bidResponse = modelAdapter.buildResponse(response);
            }
        }
        if (null != this.bidResponse) {
            if (StringUtils.isEmpty(bidResponse.getId())) {
                bidResponse.setId(bidRequest.getId());
            }
            // FIXME: 2020/2/24 以下循环的逻辑因为dsp不支持返回tagid和cid
            if (!CollectionUtils.isEmpty(bidResponse.getSeat_bid())) {
                List<Bid> bidList = bidResponse.getSeat_bid().stream()
                        .filter(seatBid -> !CollectionUtils.isEmpty(seatBid.getBid()))
                        .map(SeatBid::getBid).flatMap(Collection::stream).collect(Collectors.toList());
                if (CollectionUtils.isEmpty(bidList)) {
                    return null;
                }
                for (Bid bid : bidList) {
                    if (!isLegalBid(bid)) {
                        if (isOp()) {
                            //如果是广告优化dsp返回的广告，只需要广告有曝光链接即可
                        } else {
                            return null;
                        }
                    }
                    if (StringUtils.isEmpty(bid.getTag_id())) {
                        bid.setTag_id(adxRequest.getImps().get(0).getTag_id());
                    }
                    if (StringUtils.isEmpty(bid.getCreate_id())) {
                        bid.setCreate_id(RandomStringUtils.randomAlphabetic(6));
                    }
                }
            } else {
                return null;
            }
        }
        return bidResponse;
    }

    /**
     * 判断是否是广告优化dsp，广告优化dsp不需要返回正常广告，只需要用到其返回的曝光链接
     * @return
     */
    public boolean isOp() {
        String code = buyer.getCode();
        if ("optimization".equalsIgnoreCase(code)) {
            return true;
        }
        return false;
    }

    private boolean isLegalBid(Bid bid) {
        if (Objects.isNull(bid)) {
            return false;
        }

        if (Objects.isNull(bid.getA_native()) || CollectionUtils.isEmpty(bid.getA_native().getAssets())) {
            if (StringUtils.isEmpty(bid.getClick_through_url())
                    && StringUtils.isEmpty(bid.getApp_download_url())
                    && StringUtils.isEmpty(bid.getDeep_link_url())) {
                return false;
            }
        }
        return true;
    }

    public Buyer getBuyer() {
        return buyer;
    }

    public void setBuyer(Buyer buyer) {
        this.buyer = buyer;
    }

    public byte[] getRequest() {
        return request;
    }

    public void setRequest(byte[] request) {
        this.request = request;
    }

    public byte[] getResponse() {
        return response;
    }

    public void setResponse(byte[] response) {
        this.response = response;
    }

    public BidRequest getBidRequest() {
        return bidRequest;
    }

    public AdxErrEnum getAdxErrEnum() {
        return adxErrEnum;
    }

    public BidRequest getAdxRequest() {
        return adxRequest;
    }
}
