package com.bxm.adx.common.market.filter;

import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.market.Deal;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.BidResponse;
import com.bxm.adx.common.sell.response.Bid;
import com.bxm.adx.common.sell.response.SeatBid;
import com.bxm.adx.common.utils.MapHelper;
import com.bxm.mcssp.common.enums.app.DockingMethodTypeEnum;
import com.bxm.warcar.utils.JsonHelper;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;

import java.math.BigDecimal;
import java.util.*;

/**
 * 低价过滤器，同一个广告位只保留出价最高的。
 * 只支持单个seatbid
 *
 * @author allen
 * @since 2019-12-18
 */
@Order(Ordered.LOWEST_PRECEDENCE)
@Slf4j
@Configuration
public class PriceLowerFilter implements Filter {

    @Override
    public void filter(List<Deal> deals) {
        //获取最高cpm最高出价
        Optional<Bid> topPriceBid = deals.stream()
                .filter(deal -> Objects.nonNull(deal.getBidResponse()))
                .map(Deal::getBidResponse)
                .filter(bidResponse -> CollectionUtils.isNotEmpty(bidResponse.getSeat_bid()))
                .map(BidResponse::getSeat_bid)
                .flatMap(Collection::stream)
                .filter(seatBid -> CollectionUtils.isNotEmpty(seatBid.getBid()))
                .map(SeatBid::getBid)
                .flatMap(Collection::stream)
                .filter(bid -> Objects.nonNull(bid.getPrice()))
                //按照价格从大到小排列取第一个
                .sorted(Comparator.comparing(Bid::getPrice).reversed()).findFirst();
        if (!topPriceBid.isPresent()) {
            if (log.isDebugEnabled()) {
                log.debug("No valid price");
            }
            //找不到正常出价，排除所有deal
            deals.removeIf(deal -> {
                return true;
            });
            return;
        }
        final BigDecimal topPrice = topPriceBid.get().getPrice();

        Map<String, BigDecimal> higher = Maps.newHashMap();
        String key = "higher";
        deals.removeIf(deal -> {
            BidRequest request = deal.getBidRequest();
            BidResponse response = deal.getBidResponse();
            if (null == response) {
                return true;
            }

            List<SeatBid> seatBids = response.getSeat_bid();
            if (CollectionUtils.isEmpty(seatBids)) {
                return true;
            }
            seatBids.removeIf(seatBid -> {
                List<Bid> bidList = seatBid.getBid();
                if (CollectionUtils.isEmpty(bidList)) {
                    return true;
                }

                bidList.removeIf(b -> {
                    BigDecimal price = b.getPrice();
                    if (null == price) {
                        return true;
                    }
                    //cpc模式下暂不过滤,后续还需改动
                    if (isCpc(request)) {
                        return false;
                    }
                    //api接入模式下需要进行底价过滤
                    BigDecimal bidFloor = null;
                    if (isApi(request)) {
                        bidFloor = request.getImps().iterator().next().getBid_floor();
                    }
                    if (Objects.nonNull(bidFloor)) {
                        if (price.compareTo(bidFloor) < 0) {
                            return true;
                        }
                    }

                    //比价逻辑: 比最高价格小的直接过滤;和最高价相同的取第一个取到的bid，其他都过滤;比最高价高的输出告警
                    if (price.compareTo(topPrice) < 0) {
                        return true;
                    } else if (price.compareTo(topPrice) == 0) {
                        if (org.springframework.util.CollectionUtils.isEmpty(higher)) {
                            MapHelper.get(higher, key, price);
                            return false;
                        } else {
                            return true;
                        }
                    } else {
                        log.warn("warning bid top price wrong, top price = {},  deal = {}", topPrice.toString(), JsonHelper.convert(deal));
                        MapHelper.get(higher, key, price);
                        return false;
                    }
                });
                return CollectionUtils.isEmpty(bidList);
            });
            return CollectionUtils.isEmpty(seatBids);
        });
    }

    @Override
    public int getOrder() {
        return Ordered.LOWEST_PRECEDENCE;
    }

    private boolean isApi(BidRequest request) {
        if (Objects.nonNull(request.getDockingMethodType())
                && DockingMethodTypeEnum.API_OPERATION.getType().equals(request.getDockingMethodType())) {
            return true;
        }
        return false;
    }

    private boolean isCpc(BidRequest request) {
        int bidModel = Optional.ofNullable(request.getBid_model()).orElse(AdxConstants.BidModel.SUPPORT_CPM);
        if (AdxConstants.BidModel.SUPPORT_CPC == bidModel) {
            return true;
        }
        return false;
    }
}
