package com.bxm.adx.common.micrometer;

import com.bxm.adx.common.buy.Buyer;
import com.google.common.collect.Maps;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.Timer;
import io.micrometer.core.instrument.binder.MeterBinder;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationListener;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.util.ClassUtils;

import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * @author allen
 * @date 2020-11-02
 * @since 1.0
 */
@Slf4j
@Configuration
public class BuyerMeter implements MeterBinder, ApplicationListener<ContextRefreshedEvent> {

    private final Map<String, Timer> requestTimer = Maps.newHashMap();
    private final Map<String, Counter> requestCounter = Maps.newHashMap();
    private final Map<String, Counter> paddedCounter = Maps.newHashMap();
    private final Map<String, Counter> connectExceptionCounter = Maps.newHashMap();
    private final Map<String, Counter> readExceptionCounter = Maps.newHashMap();
    private final Map<String, Counter> httpStatusUn2XXCounter = Maps.newHashMap();

    private MeterRegistry registry;

    /**
     * 记录请求
     * @param bean 对象
     * @param start 开始时间
     */
    public void recordRequest(Object bean, long start) {
        String key = getKey(bean);
        Timer timer = requestTimer.get(key);
        if (Objects.nonNull(timer)) {
            timer.record((System.nanoTime() - start), TimeUnit.NANOSECONDS);
        }
    }

    /**
     * 请求增长
     * @param bean bean
     */
    public void increaseRequest(Object bean) {
        String key = getKey(bean);
        Counter counter = requestCounter.get(key);
        if (Objects.nonNull(counter)) {
            counter.increment();
        }
    }

    /**
     * 填充增长
     * @param bean bean
     */
    public void increasePadding(Object bean) {
        String key = getKey(bean);
        Counter counter = paddedCounter.get(key);
        if (Objects.nonNull(counter)) {
            counter.increment();
        }
    }

    /**
     * 异常增长
     * @param bean bean
     */
    public void increaseConnectException(Object bean) {
        String key = getKey(bean);
        Counter counter = connectExceptionCounter.get(key);
        if (Objects.nonNull(counter)) {
            counter.increment();
        }
    }

    /**
     * 异常增长
     * @param bean bean
     */
    public void increaseReadException(Object bean) {
        String key = getKey(bean);
        Counter counter = readExceptionCounter.get(key);
        if (Objects.nonNull(counter)) {
            counter.increment();
        }
    }

    public void increaseHttpStatusUn2XX(Object bean) {
        String key = getKey(bean);
        Counter counter = httpStatusUn2XXCounter.get(key);
        if (Objects.nonNull(counter)) {
            counter.increment();
        }
    }

    @Override
    public void onApplicationEvent(ContextRefreshedEvent event) {
        ApplicationContext applicationContext = event.getApplicationContext();
        Map<String, Buyer> beans = applicationContext.getBeansOfType(Buyer.class);
        for (Map.Entry<String, Buyer> entry : beans.entrySet()) {
            Buyer buyer = entry.getValue();
            bindTo(buyer);
        }
    }

    @Override
    public void bindTo(MeterRegistry registry) {
        this.registry = registry;
    }

    private void bindTo(Buyer buyer) {
        String key = getKey(buyer);
        bindRequestTimer(key);
        bindRequestCounterMap(key);
        bindPaddedCounterMap(key);
        bindConnectExceptionCounterMap(key);
        bindReadExceptionCounterMap(key);
        bindHttpStatusUn2XXCounterMap(key);
    }

    private void bindRequestCounterMap(String key) {
        if (requestCounter.containsKey(key)) {
            return;
        }
        Counter counter = Counter.builder("adx.buyer.request").tags("name", key).register(registry);
        requestCounter.put(key, counter);
    }

    private void bindPaddedCounterMap(String key) {
        if (paddedCounter.containsKey(key)) {
            return;
        }
        Counter counter = Counter.builder("adx.buyer.padding").tags("name", key).register(registry);
        paddedCounter.put(key, counter);
    }

    private void bindConnectExceptionCounterMap(String key) {
        if (connectExceptionCounter.containsKey(key)) {
            return;
        }
        Counter counter = Counter.builder("adx.buyer.connect_exception").tags("name", key).register(registry);
        connectExceptionCounter.put(key, counter);
    }

    private void bindReadExceptionCounterMap(String key) {
        if (readExceptionCounter.containsKey(key)) {
            return;
        }
        Counter counter = Counter.builder("adx.buyer.read_exception").tags("name", key).register(registry);
        readExceptionCounter.put(key, counter);
    }

    private void bindRequestTimer(String key) {
        if (requestTimer.containsKey(key)) {
            return;
        }
        Timer timer = Timer.builder("adx.buyer").tags("name", key).register(this.registry);
        requestTimer.put(key, timer);
    }

    private void bindHttpStatusUn2XXCounterMap(String key) {
        if (httpStatusUn2XXCounter.containsKey(key)) {
            return;
        }
        Counter counter = Counter.builder("adx.buyer.http_status_err").tags("name", key).register(registry);
        httpStatusUn2XXCounter.put(key, counter);
    }

    private String getKey(Object bean) {
        return ClassUtils.getUserClass(bean).getName();
    }
}
