package com.bxm.adx.common.openlog.listener.external;

import com.bxm.adx.common.AdxProperties;
import com.bxm.adx.common.DotEventControlProperties;
import com.bxm.adx.common.OpenlogConstants;
import com.bxm.adx.common.openlog.event.external.AdShowEvent;
import com.bxm.adx.facade.constant.redis.AdxKeyGenerator;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.openlog.sdk.consts.Adx;
import com.bxm.warcar.cache.Counter;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.integration.eventbus.EventListener;
import com.bxm.warcar.integration.eventbus.core.AllowConcurrentEvents;
import com.bxm.warcar.integration.eventbus.core.Subscribe;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.context.annotation.Configuration;

import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * 通过埋点事件单位时间内次数上限来控制是否请求dsp
 * @author fgf
 * @date 2022-11-21
 * @since 1.0
 */
@Slf4j
@Configuration
public class AdShowForDspAvgExposureLimiterEventListener implements EventListener<AdShowEvent> {

    private final Counter counter;
    private final AdxProperties adxProperties;
    private final int EXPIRE_SECONDS_12_HOURS = 12 * 3600;
    private final int EXPIRE_SECONDS_24_HOURS = 24 * 3600;

    public AdShowForDspAvgExposureLimiterEventListener(Counter counter, AdxProperties adxProperties) {
        this.counter = counter;
        this.adxProperties = adxProperties;
    }

    @Override
    @Subscribe
    @AllowConcurrentEvents
    public void consume(AdShowEvent event) {
        handleDspAvgExposure(event);
    }

    private void handleDspAvgExposure(AdShowEvent event) {
        KeyValueMap map = event.getLog();
        String dspId = map.getFirst(Adx.Param.DSPID);
        String dspTagId = map.getFirst(Adx.Param.DTAGID);

        LocalDateTime now = LocalDateTime.now();
        DotEventControlProperties control = needControl(dspId, dspTagId, now);
        if (Objects.isNull(control)) {
            return;
        }

        KeyGenerator sectionKey = AdxKeyGenerator.Counter.getDspPositionExposureBySection(control.getId(), control.getSectionId(now),
                dspId, dspTagId);

        KeyGenerator controlKey = AdxKeyGenerator.Counter.getDspPositionExposureByControl(control.getId(), dspId, dspTagId);

        counter.incrementAndGet(sectionKey, EXPIRE_SECONDS_12_HOURS);
        counter.incrementAndGet(controlKey, EXPIRE_SECONDS_24_HOURS);
    }

    private DotEventControlProperties needControl(String dspId, String dspTagId, LocalDateTime now) {
        List<DotEventControlProperties> properties = adxProperties.getDotEventControls();
        if (CollectionUtils.isEmpty(properties)) {
            return null;
        }

        Optional<DotEventControlProperties>  optional = properties.stream()
                .filter(p -> Objects.nonNull(p.getMt()) && OpenlogConstants.Mt.IMP_MT == p.getMt())
                .filter(p -> p.getDspId().equals(dspId))
                .filter(p -> p.getDspPositionId().equals(dspTagId))
                .filter(p -> now.isAfter(p.getStart()) && now.isBefore(p.getEnd()))
                .findFirst();
        return optional.isPresent() ?  optional.get() : null;
    }
}
