package com.bxm.adx.common.sell.position.cached;

import com.bxm.adx.common.AdxProperties;
import com.bxm.adx.common.sell.position.Position;
import com.bxm.adx.common.sell.position.PositionChangeHandler;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.context.annotation.Configuration;

import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @author fgf
 * @date 2022/12/14
 **/
@Configuration
@Slf4j
public class PositionCached implements PositionChangeHandler {
    private final AdxProperties properties;
    /**
     * 广告位ID-广告位集合
     * <pre>
     * |-------------|-----------------------------|
     * |    Key      | Value                       |
     * |-------------|-----------------------------|
     * | {positionId}|[position,position,...]      |
     * |-------------|-----------------------------|
     * </pre>
     */
    private final ConcurrentHashMap<String, Position> positionIdMap = new ConcurrentHashMap<>();
    /**
     * 媒体广告位ID-广告位集合
     * <pre>
     * |--------------------|-----------------------------|
     * |    Key             | Value                       |
     * |--------------------|-----------------------------|
     * | {mediaPositionId}  |[position,position,...]      |
     * |--------------------|-----------------------------|
     * </pre>
     */
    private final ConcurrentHashMap<String, Position> mediaPositionIdMap = new ConcurrentHashMap<>();
    /**
     * 广告位尺寸类型-广告位集合
     * <pre>
     * |----------------------------|-----------------------------|
     * |    Key                     | Value                       |
     * |----------------------------|-----------------------------|
     * | {appId-positionScene-w*h}  |[position,position,...]      |
     * |----------------------------|-----------------------------|
     * </pre>
     */
    private final ConcurrentHashMap<String, Position> positionInfoMap = new ConcurrentHashMap<>();

    public PositionCached(AdxProperties properties) {
        this.properties = properties;
    }

    @Override
    public void doUpdate(Position old, Position latest) {
        if (isExecuteUpdateAndDoDelete(old, latest)) {

            positionIdMap.put(latest.getPositionId(), latest);

            String mediaPositionId = latest.getAppPositionId();
            if (StringUtils.isNotEmpty(mediaPositionId)) {
                mediaPositionIdMap.put(mediaPositionId, latest);
            }
            if (isBesPosition(latest)) {
                String infoKey = buildMapKeyByPosition(latest);
                if (StringUtils.isNotEmpty(infoKey)) {
                    positionInfoMap.put(infoKey, latest);
                }
            }
        }
    }

    @Override
    public void doDelete(Position old) {
        positionIdMap.remove(old.getPositionId());
        String mediaPositionId = old.getAppPositionId();
        if (StringUtils.isNotEmpty(mediaPositionId)) {
            mediaPositionIdMap.remove(mediaPositionId);
        }
        if (isBesPosition(old)) {
            String infoKey = buildMapKeyByPosition(old);
            if (StringUtils.isNotEmpty(infoKey)) {
                positionInfoMap.remove(infoKey);
            }
        }
    }

    /**
     * 根据bxm广告位id/媒体广告位id获取广告位
     * @param tagId
     * @return
     */
    public Position getPositionByTagId(String tagId) {
        return Optional.ofNullable(positionIdMap.get(tagId)).orElse(mediaPositionIdMap.get(tagId));
    }

    /**
     * 根据bxm广告位id获取广告位
     * @param positionId
     * @return
     */
    public Position getPositionById(String positionId){
        return positionIdMap.get(positionId);
    }

    /**
     * 根据媒体广告位ID获取广告位
     * @param positionId
     * @return
     */
    public Position getPositionByAppTagId(String positionId){
        return mediaPositionIdMap.get(positionId);
    }
    /**
     * 获取复杂广告位映射（比如bes）
     * @param appId
     * @param w
     * @param h
     * @param positionScene
     * @return
     */
    public Position getPositionByInfo(String appId, Integer w, Integer h, Integer positionScene){
        if (StringUtils.isEmpty(appId)) {
            return null;
        }
        if (Objects.isNull(w) || Objects.isNull(h)) {
            return null;
        }
        if (Objects.isNull(positionScene)) {
            return null;
        }
        String key = buildMapKey(appId, w, h, positionScene);
        return positionInfoMap.get(key);
    }

    /**
     * 构建复杂广告位映射keu（比如bes）
     * @param position
     * @return
     */
    public String buildMapKeyByPosition(Position position) {
        String appId = position.getAppId();
        String positionSize = position.getPositionSize();
        Byte scene = position.getPositionScene();
        if (StringUtils.isEmpty(appId)) {
            return null;
        }
        if (StringUtils.isEmpty(positionSize)) {
            return null;
        }
        if (Objects.isNull(scene)) {
            return null;
        }
        String size = positionSize.split(" ")[0];
        String val = new StringBuilder(appId)
                .append("-")
                .append(position.getPositionScene())
                .append("-")
                .append(size)
                .toString();
        return val;
    }

    /**
     * 构建复杂广告位映射keu（比如bes）
     * @param appId
     * @param w
     * @param h
     * @param positionScene
     * @return
     */
    public String buildMapKey(String appId, int w, int h, int positionScene) {
        String val = new StringBuilder(appId)
                .append("-")
                .append(positionScene)
                .append("-")
                .append(w)
                .append("*")
                .append(h)
                .toString();
        return val;
    }

    /**
     * 是否是bes广告位
     * @param position
     * @return
     */
    private boolean isBesPosition(Position position) {
        if (!org.springframework.util.StringUtils.isEmpty(position.getPositionSize())) {
            if (position.getAppId().equals(properties.getBesAndroidAppId())) {
                return true;
            }
            if (position.getAppId().equals(properties.getBesIosAppId())) {
                return true;
            }
        }
        return false;
    }
}
