package com.bxm.adx.common.shieldstrategy;

import com.bxm.adx.facade.constant.pushable.CachePushableFields;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.cache.push.Pushable;
import com.bxm.warcar.cache.serialization.JSONSerialization;
import com.bxm.warcar.integration.pushable.annotation.CachePush;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.KeyBuilder;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.Pipeline;

import java.util.*;

/**
 *
 * For cache pushable
 *
 * <pre>
 * name = "SHIELD_STRATEGY"
 * parameters = {id=shieldStrategyId}
 * byte[] = ShieldStrategy object to json bytes
 * </pre>
 * @author zhengwangeng
 * @since 2020-1-11
 */
@Component
@CachePush("SHIELD_STRATEGY")
public class ShieldStrategyDaoImpl implements ShieldStrategyDao, Pushable {

    private final Fetcher fetcher;
    private final Updater updater;

    public ShieldStrategyDaoImpl(Fetcher fetcher, Updater updater) {
        this.fetcher = fetcher;
        this.updater = updater;
    }

    @Override
    public ShieldStrategy getById(String id) {
        return fetcher.fetch(getKeyGenerator(id), ShieldStrategy.class);
    }

    @Override
    public List<ShieldStrategy> getByIds(String shieldStrategyIds) {
        if (StringUtils.isNotBlank(shieldStrategyIds)) {
            String[] shieldStrategyIdArray = shieldStrategyIds.split(",");
            if (shieldStrategyIdArray.length > 0) {
                List<ShieldStrategy> resultList = new ArrayList<>(shieldStrategyIdArray.length);
                JSONSerialization serialization = new JSONSerialization();

                JedisPool jedisPool = (JedisPool)fetcher.getClientOriginal();
                try (Jedis jedis = jedisPool.getResource()) {
                    Pipeline pipeline = jedis.pipelined();
                    for (String shieldStrategyId : shieldStrategyIdArray) {
                        ShieldStrategy ref =
                                serialization.deserialize(pipeline.get(getKeyGenerator(shieldStrategyId).generateKey()).get(), ShieldStrategy.class);
                        resultList.add(ref);
                    }
                    pipeline.syncAndReturnAll();
                }
                return resultList;
            }
        }

        return Collections.emptyList();
    }

    @Override
    public void push(Map<String, Object> parameters, byte[] data) {
        Object o = parameters.get(CachePushableFields.SHIELD_STRATEGY_ID);

        if (Objects.isNull(o)) {
            return;
        }
        String id = Objects.toString(o);
        ShieldStrategy shieldStrategy = JsonHelper.convert(data, ShieldStrategy.class);

        updater.remove(getKeyGenerator(id));
        if (null != shieldStrategy && null != shieldStrategy.getId()) {
            updater.update(getKeyGenerator(id), shieldStrategy);
        }
    }

    private KeyGenerator getKeyGenerator(String value) {
        return () -> KeyBuilder.build("ADX", "SHIELD_STRATEGY", "ID", value);
    }
}
