package com.bxm.adx.common.adapter.aspect;

import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.adapter.BidModelAdapter;
import com.bxm.adx.common.log.*;
import com.bxm.adx.common.log.buriedSwitch.BuriedSwitchCached;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.position.Position;
import com.bxm.adx.common.sell.position.mapping.PositionMappingStrategyFactory;
import com.bxm.adx.common.sell.request.Impression;
import com.bxm.warcar.utils.JsonHelper;
import lombok.extern.slf4j.Slf4j;
import org.aopalliance.intercept.MethodInterceptor;
import org.aopalliance.intercept.MethodInvocation;

import java.util.Base64;
import java.util.Objects;

/**
 * @author fgf
 * @date 2024/3/4
 **/
@Slf4j
public class ModelAdapterRequestLogAdvice implements MethodInterceptor {
    private final ByteLogger byteLogger;
    private final PositionMappingStrategyFactory positionMappingStrategyFactory;
    private final BuriedSwitchCached buriedSwitchCached;
    private final AdxDataLogMonitor adxDataLogMonitor;
    private final DailyLogService dailyLogService;

    public ModelAdapterRequestLogAdvice(ByteLogger byteLogger, PositionMappingStrategyFactory positionMappingStrategyFactory,
                                        BuriedSwitchCached buriedSwitchCached, AdxDataLogMonitor adxDataLogMonitor, DailyLogService dailyLogService) {
        this.byteLogger = byteLogger;
        this.positionMappingStrategyFactory = positionMappingStrategyFactory;
        this.buriedSwitchCached = buriedSwitchCached;
        this.adxDataLogMonitor = adxDataLogMonitor;
        this.dailyLogService = dailyLogService;
    }

    @Override
    public Object invoke(MethodInvocation invocation) throws Throwable {
        Object[] arguments = invocation.getArguments();
        if (arguments.length != 1 && !(arguments[0] instanceof byte[])) {
            return invocation.proceed();
        }

        byte[] data = (byte[]) arguments[0];

        Object invocationThis = invocation.getThis();
        if (!(invocationThis instanceof BidModelAdapter)) {
            return invocation.proceed();
        }

        BidModelAdapter target = (BidModelAdapter) invocationThis;
        Object result = invocation.proceed();
        BidRequest bidRequest = (BidRequest) result;
        Position position = getPosition(bidRequest);
        if (Objects.nonNull(position)) {
            //全埋点功能
            if (checkBury(position)) {
                adxDataLogMonitor.requestMonitor(position.getPositionId());
                try {
                    byteLogger.add(buildLog(target.getRequestFormat(), data, LogSourceEnum.MediaRequest, position, bidRequest));
                    byteLogger.add(buildLog(AdxConstants.Format.Json, JsonHelper.convert2bytes(result), LogSourceEnum.BxmRequest, position, bidRequest));
                } catch (Exception e) {
                    log.warn("add: ", e);
                }
            }
            //日常媒体请求日志记录（bxm协议）
            dailyLogService.log(buildLog(AdxConstants.Format.Json, JsonHelper.convert2bytes(result), LogSourceEnum.BxmRequest, position, bidRequest));
        }
        return result;
    }

    private AdxDataLog buildLog(AdxConstants.Format format, byte[] data, LogSourceEnum source, Position position, BidRequest bidRequest) {
        String request = null;
        switch (format) {
            case Protobuf:
                request = Base64.getEncoder().encodeToString(data);
                break;
            case Json:
                request = new String(data);
                break;
        }
        return AdxDataLog.builder()
                .bidId(bidRequest.getId())
                .positionId(position.getPositionId())
                .format(format)
                .request(request)
                .source(source)
                .build();
    }

    private boolean checkBury(Position position) {
        String positionId = position.getPositionId();
        return buriedSwitchCached.contains(positionId);
    }

    private Position getPosition(BidRequest bidRequest) {
        try {
            Impression impression = bidRequest.getImps().iterator().next();
            //映射广告位
            Position mediaPosition = positionMappingStrategyFactory.getPosition(bidRequest, impression);
            return mediaPosition;
        } catch (Exception e) {
            return null;
        }
    }
}
