package com.bxm.adx.common.market.exchange.rebuild.request;

import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.buy.dispatcher.Dispatcher;
import com.bxm.adx.common.buy.dispatcher.DispatcherPriceConfig;
import com.bxm.adx.common.buy.position.AdvertPoint;
import com.bxm.adx.common.buy.position.AdvertPointService;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.facade.constant.enums.AdxErrEnum;
import com.bxm.adx.facade.exception.AdxException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.annotation.Configuration;

import java.math.BigDecimal;
import java.util.Objects;
import java.util.Optional;

/**
 * @author fgf
 * @date 2023/1/9
 **/
@Slf4j
@Configuration
public class RequestBuilder implements AdxBidRequestBuilder {
    private final AdvertPointService advertPointService;

    public RequestBuilder(AdvertPointService advertPointService) {
        this.advertPointService = advertPointService;
    }

    @Override
    public void rebuildAdxBidRequest(BidRequest adxBidRequest, RequestBuildAttribute attribute) {
        adxBidRequest.setBid_model(getBidModel(attribute));
    }


    /**
     * 根据dsp广告位结算类型判断对dsp的竞价模式
     * cpc结算则cpc竞价
     * cpm结算则cpm竞价
     *
     * @param attribute
     * @return
     */
    private Integer getBidModel(RequestBuildAttribute attribute) {
        Dispatcher dispatcher = attribute.getBuyerWrapper().getDispatcher();
        AdvertPoint advertPoint = advertPointService.getAdvertPointByDspId(dispatcher.getDspId().toString(),
                dispatcher.getDspAppid(), dispatcher.getDspPosid());
        if (Objects.isNull(advertPoint)) {
            log.error("dispatcher {} err", dispatcher.getId());
            throw new AdxException(AdxErrEnum.DISPATCHER_ERR);
        }
        Integer costPerType = Optional.ofNullable(advertPoint.getCostPerType()).orElse(AdxConstants.CostType.CPM);
        if (Objects.nonNull(costPerType)) {
            switch (costPerType) {
                case AdxConstants.AdPointCostType.CPM:
                    return AdxConstants.BidModel.SUPPORT_CPM;
                case AdxConstants.AdPointCostType.CPC:
                    return AdxConstants.BidModel.SUPPORT_CPC;
                default:
                    log.error("unsupport type {} err", costPerType);
                    throw new AdxException(AdxErrEnum.DISPATCHER_ERR);
            }
        }
        log.warn("advert-point {} cost-type is empty", advertPoint.getUniqueKey());
        return AdxConstants.BidModel.SUPPORT_CPM;
    }
}
