package com.bxm.adx.common.market.filter;

import com.bxm.adx.common.market.Deal;
import com.bxm.adx.common.sell.BidResponse;
import com.bxm.adx.common.sell.response.Bid;
import com.bxm.adx.common.sell.response.SeatBid;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;

import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import static com.bxm.adx.common.AdxConstants.ChargeType;
import static com.bxm.adx.common.AdxConstants.BidModel;

/**
 * 计费类型过滤，cpc返回cpc，cpm返回cpm，ocpm 计费类型不过滤
 *
 * @author lijingyu
 * @since 2024-01-22
 */
@Slf4j
@Configuration
public class BidModelFilter implements Filter {

    @Override
    public void filter(List<Deal> deals, Set<Deal> trash) {
        Set<Deal> remove = deals.stream().filter(
                deal -> {
                    Integer bidModel = deal.getBidRequest().getBid_model();
                    BidResponse response = deal.getBidResponse();

                    List<SeatBid> seatBids = response.getSeat_bid();
                    seatBids.removeIf(seatBid -> {
                        List<Bid> bidList = seatBid.getBid();
                        bidList.removeIf(b ->
                                {
                                    int chargeType = b.getCharge_type();
                                    return checkBidChargeType(chargeType, bidModel);
                                }
                        );

                        return CollectionUtils.isEmpty(bidList);
                    });
                    return CollectionUtils.isEmpty(seatBids);
                }
        ).collect(Collectors.toSet());
        if (CollectionUtils.isNotEmpty(remove)) {
            trash.addAll(remove);
        }
    }

    /**
     * bidModel=CPM 只能出 CPM
     * bidModel=CPC 只能出 CPC
     * bidModel=CPM_CPC 能出 CPM和CPC
     * bidModel=CPM_CPC_OCPM 能出 CPM、CPC和OCPM
     * 返回 true 代表被过滤掉
     */
    private boolean checkBidChargeType(int chargeType, int bidModel) {
        switch (bidModel) {
            case BidModel.SUPPORT_CPM:
                return chargeType != ChargeType.CPM;

            case BidModel.SUPPORT_CPC:
                return chargeType != ChargeType.CPC;

            case BidModel.CPM_CPC:
                return chargeType != ChargeType.CPM && chargeType != ChargeType.CPC;

            case BidModel.CPM_CPC_OCPM:
                return chargeType != ChargeType.CPM && chargeType != ChargeType.CPC && chargeType != ChargeType.OCPM;

            default:
                return true;
        }
    }
    @Override
    public int getOrder() {
        return Ordered.LOWEST_PRECEDENCE - 2;
    }
}
