package com.bxm.adx.common.market.filter;

import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.AdxProperties;
import com.bxm.adx.common.adapter.AdxContextFactory;
import com.bxm.adx.common.market.Deal;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.BidResponse;
import com.bxm.adx.common.sell.response.Bid;
import com.bxm.adx.common.sell.response.SeatBid;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.context.annotation.Configuration;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 最多出2个OCPM
 *
 * @author fgf
 * @date 2024/9/13
 **/
@Slf4j
@Configuration
public class OcpmPriceFilter implements Filter {
    private final AdxProperties properties;

    public OcpmPriceFilter(AdxProperties properties) {
        this.properties = properties;
    }

    @Override
    public void filter(List<Deal> deals, Set<Deal> trash) {
        BidRequest request = AdxContextFactory.get().getBidRequest();
        int bidModel = request.getBid_model();
        if (AdxConstants.BidModel.CPM_CPC_OCPM != bidModel) {
            return;
        }

        List<Bid> bidList = deals.stream()
                .filter(deal -> Objects.nonNull(deal.getBidResponse()))
                .map(Deal::getBidResponse)
                .filter(bidResponse -> CollectionUtils.isNotEmpty(bidResponse.getSeat_bid()))
                .map(BidResponse::getSeat_bid)
                .flatMap(Collection::stream)
                .filter(seatBid -> CollectionUtils.isNotEmpty(seatBid.getBid()))
                .map(SeatBid::getBid)
                .flatMap(Collection::stream)
                .filter(bid -> AdxConstants.ChargeType.OCPM == bid.getCharge_type() && Objects.nonNull(bid.getObid()))
                .sorted(Comparator.comparing(Bid::getObid).reversed()).collect(Collectors.toList());
        if (CollectionUtils.isEmpty(bidList) || bidList.size() <= properties.getMaxCreativesOfOcpm()) {
            return;
        }
        //校验bid是否重复
        Set<Bid> bidSet = new HashSet<>(bidList);
        if (bidSet.size() != bidList.size()) {
            log.warn("bid id&adid is duplicate");
        }

        List<Bid> removeList = new ArrayList<>(bidList).subList(properties.getMaxCreativesOfOcpm(), bidList.size());
        Set<Deal> remove = deals.stream().filter(
                deal -> {
                    BidResponse response = deal.getBidResponse();

                    List<SeatBid> seatBids = response.getSeat_bid();
                    seatBids.removeIf(seatBid -> {
                        List<Bid> bids = seatBid.getBid();

                        bids.removeIf(b -> {
                            return removeList.contains(b);
                        });
                        return CollectionUtils.isEmpty(bids);
                    });
                    return CollectionUtils.isEmpty(seatBids);
                }
        ).collect(Collectors.toSet());
        if (CollectionUtils.isNotEmpty(remove)) {
            trash.addAll(remove);
        }

    }

    @Override
    public int getOrder() {
        return 0;
    }
}
