package com.bxm.adx.common.openlog.listener.internal;

import com.bxm.adx.common.AdxProperties;
import com.bxm.adx.common.OpenlogConstants;
import com.bxm.adx.common.openlog.event.internal.DspBidEvent;
import com.bxm.adx.common.sell.BidResponse;
import com.bxm.adx.common.sell.builder.BuildAttribute;
import com.bxm.adx.common.sell.builder.BuildAttributeStringMacrosHandler;
import com.bxm.adx.common.sell.response.Bid;
import com.bxm.adx.common.sell.response.SeatBid;
import com.bxm.openlog.extension.client.OpenLogClient;
import com.bxm.warcar.integration.eventbus.EventListener;
import com.bxm.warcar.integration.eventbus.core.AllowConcurrentEvents;
import com.bxm.warcar.integration.eventbus.core.Subscribe;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.UrlHelper;
import lombok.Data;
import lombok.experimental.Accessors;
import org.apache.commons.collections.MapUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.CollectionUtils;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * dsp高质量返回设备号人群包埋点
 * @author lijingyu
 * @date 2024/5/15
 **/
@Configuration
public class DotDspBidForUslEventListener implements EventListener<DspBidEvent> {

    private OpenLogClient openLogClient;
    private final AdxProperties properties;
    private final BuildAttributeStringMacrosHandler macrosHandler;

    public DotDspBidForUslEventListener(AdxProperties properties, BuildAttributeStringMacrosHandler macrosHandler) {
        this.properties = properties;
        this.macrosHandler = macrosHandler;
    }

    @Autowired
    public void setOpenLogClient(OpenLogClient openLogClient) {
        this.openLogClient = openLogClient;
    }

    @Override
    @Subscribe
    @AllowConcurrentEvents
    public void consume(DspBidEvent event) {
        MultiValueMap<String, String> customizeParams = buildCustomizeParams(event.getDspResponse());
        if (MapUtils.isEmpty(customizeParams)) {
            return;
        }

        String url = properties.getOpenLog().createNeed(true, OpenlogConstants.Mt.DSP_BID_USL,customizeParams,OpenlogConstants.DSP_BID_USL_WITH_MACROS);
        BuildAttribute attribute = new BuildAttribute(OpenlogConstants.Mt.DSP_RESPONSE_MT)
                .setSspRequest(event.getSspRequest());
        url = macrosHandler.replaceAll(url, attribute);
        openLogClient.asyncRequest(url);
    }

    /**
     * 构建客制化埋点参数
     * @param bidResponse dsp响应
     * @return 包装好的质量分
     */
    private MultiValueMap<String, String> buildCustomizeParams(BidResponse bidResponse) {
        List<DotDspBidForUslEventListener.Creative> creatives = getCreatives(bidResponse);
        if (CollectionUtils.isEmpty(creatives)) {
            return null;
        }
        MultiValueMap<String, String> customizeParams = new LinkedMultiValueMap<>();
        String value = UrlHelper.urlEncode(JsonHelper.convert(creatives));
        customizeParams.add(OpenlogConstants.MacrosCustomizeParams.BID_CREATIVES, value);
        return customizeParams;
    }

    /**
     * 参竞广告详情
     * @param bidResponse dsp响应
     * @return 质量分
     */
    private List<DotDspBidForUslEventListener.Creative> getCreatives(BidResponse bidResponse) {
        List<DotDspBidForUslEventListener.Creative> creatives = new ArrayList<>();
        for (SeatBid seatBid : bidResponse.getSeat_bid()) {
            for (Bid bid : seatBid.getBid()) {
                Integer userScoreLevel = bid.getUser_score_level();
                if (Objects.isNull(userScoreLevel) || userScoreLevel < 80) {
                    continue;
                }
                DotDspBidForUslEventListener.Creative creative = new DotDspBidForUslEventListener.Creative()
                        .setUsl(bid.getUser_score_level());
                creatives.add(creative);
            }
        }
        return creatives;
    }

    @Data
    @Accessors(chain = true)
    private static class Creative {
        private Integer usl;
    }
}
