package com.bxm.adx.common.buy.dispatcher.filter;

import cn.hutool.core.collection.ConcurrentHashSet;
import com.bxm.adx.common.CacheKeys;
import com.bxm.adx.common.buy.dispatcher.Dispatcher;
import com.bxm.adx.common.buy.dispatcher.DispatcherContext;
import com.bxm.adx.common.buy.dispatcher.abtest.DispatcherABConfig;
import com.bxm.adx.common.buy.dispatcher.abtest.DispatcherABConfigChangeHandler;
import com.bxm.adx.common.buy.dispatcher.abtest.DispatcherConfig;
import com.bxm.adx.common.limiter.DeleteExpiredHandler;
import com.bxm.adx.common.rule.Rule;
import com.bxm.adx.common.rule.WhiteAndBlackRule;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.utils.MapHelper;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @author fgf
 * @date 2023/3/13
 **/
@Slf4j
public abstract class AbstractCrowdPackageFilter implements DispatcherFilter<Dispatcher>, DispatcherABConfigChangeHandler {
    protected Fetcher fetcher;
    protected DeleteExpiredHandler deleteExpiredHandler;

    public AbstractCrowdPackageFilter(Fetcher fetcher, DeleteExpiredHandler deleteExpiredHandler) {
        this.fetcher = fetcher;
        this.deleteExpiredHandler = deleteExpiredHandler;
    }

    /**
     * 流量分配设置的人群包id(包含黑名单)
     * <pre>
     * |----------------|-----------------------------|
     * | Key            | Value                       |
     * |----------------|-----------------------------|
     * | {dispatcherId} | [ crowdId, crowdId ]        |
     * |----------------|-----------------------------|
     * </pre>
     */
    protected final ConcurrentHashMap<String, Set<String>> dispatcherBlackCrowdIds = new ConcurrentHashMap<>();
    /**
     * 流量分配设置的人群包id(包含黑名单)
     * <pre>
     * |----------------|-----------------------------|
     * | Key            | Value                       |
     * |----------------|-----------------------------|
     * | {dispatcherId} | [ crowdId, crowdId ]        |
     * |----------------|-----------------------------|
     * </pre>
     */
    protected final ConcurrentHashMap<String, Set<String>> dispatcherWhiteCrowdIds = new ConcurrentHashMap<>();
    /**
     * 流量分配设置的人群包id(包含黑白名单)
     * <pre>
     * |----------------|-----------------------------|
     * | Key            | Value                       |
     * |----------------|-----------------------------|
     * | {dispatcherId} | [ crowdId, crowdId ]        |
     * |----------------|-----------------------------|
     * </pre>
     */
    protected final ConcurrentHashMap<String, Set<String>> dispatcherCrowdIds = new ConcurrentHashMap<>();

    @Override
    public void doUpdate(DispatcherABConfig old, DispatcherABConfig latest) {
        if (isExecuteUpdateAndDoDelete(old, latest)) {
            List<DispatcherConfig> dispatcherConfigs = latest.getDispatcherConfigCaches();
            if (CollectionUtils.isEmpty(dispatcherConfigs)) {
                return;
            }
            for (DispatcherConfig dispatcherConfig : dispatcherConfigs) {
                List<Dispatcher> dispatchers = dispatcherConfig.getDispatcherDspCaches();
                if (CollectionUtils.isEmpty(dispatchers)) {
                    continue;
                }
                for (Dispatcher dispatcher : dispatchers) {
                    if (dispatcher.getOpened() == Dispatcher.DISPATCHER_OPENED_NO) {
                        continue;
                    }
                    Rule rule = dispatcher.getCrowdPackageOrientation();
                    String id = dispatcher.getId();
                    if (Objects.isNull(rule)) {
                        continue;
                    }
                    WhiteAndBlackRule whiteAndBlackRule = new WhiteAndBlackRule(rule);
                    WhiteAndBlackRule.WhiteAndBlack whiteAndBlackSet = whiteAndBlackRule.getWhiteAndBlack();
                    MapHelper.get(dispatcherBlackCrowdIds, id, new ConcurrentHashSet<>()).addAll(whiteAndBlackSet.getBlack());
                    MapHelper.get(dispatcherWhiteCrowdIds, id, new ConcurrentHashSet<>()).addAll(whiteAndBlackSet.getWhite());
                    MapHelper.get(dispatcherCrowdIds, id, new ConcurrentHashSet<>()).addAll(whiteAndBlackSet.getAll());
                }
            }
        }
    }

    @Override
    public void doDelete(DispatcherABConfig old) {
        List<DispatcherConfig> dispatcherConfigs = old.getDispatcherConfigCaches();
        if (CollectionUtils.isEmpty(dispatcherConfigs)) {
            return;
        }
        for (DispatcherConfig dispatcherConfig : dispatcherConfigs) {
            List<Dispatcher> dispatchers = dispatcherConfig.getDispatcherDspCaches();
            if (CollectionUtils.isEmpty(dispatchers)) {
                continue;
            }
            for (Dispatcher dispatcher : dispatchers) {
                Rule rule = dispatcher.getCrowdPackageOrientation();
                if (Objects.isNull(rule)) {
                    continue;
                }
                dispatcherBlackCrowdIds.remove(dispatcher.getId());
                dispatcherWhiteCrowdIds.remove(dispatcher.getId());
                dispatcherCrowdIds.remove(dispatcher.getId());
            }
        }
    }

    /**
     * 获取用户命中的人群包id集合
     *
     * @param context
     * @return
     */
    protected Set<String> getUserCrowdPackageSet(DispatcherContext<Dispatcher> context) {
        Set<String> userCrowds = context.getUserCrowds();
        if (CollectionUtils.isNotEmpty(userCrowds)) {
            return userCrowds;
        }
        Set<String> userCrowdPackageSet = new HashSet<>();
        KeyGenerator key = CacheKeys.getCrowdPackageNewKey(context.getRequest().getDevice());
        if (Objects.nonNull(key)) {
            if (log.isDebugEnabled()) {
                log.debug("key {}", key.generateKey());
            }
            long currentSecond = System.currentTimeMillis() / 1000;
            Map<String, Long> idAndExpireSecondMap = fetcher.hfetchall(key, Long.class);
            if (MapUtils.isNotEmpty(idAndExpireSecondMap)) {
                for (Map.Entry<String, Long> entry : idAndExpireSecondMap.entrySet()) {
                    //key为人群包id，value为过期时间（时间戳秒）
                    String id = entry.getKey();
                    if (entry.getValue() > currentSecond) {
                        userCrowdPackageSet.add(id);
                    } else {
                        deleteExpiredHandler.delete(key, id);
                    }
                }
            }
        }
        context.setUserCrowds(userCrowdPackageSet);
        return userCrowdPackageSet;
    }

    /**
     * 是否命中人群包
     * @param userCrowdSet 用户命中的人群包
     * @param configSet 流量分配配置的人群包
     * @return
     */
    protected boolean hit(Set<String> userCrowdSet, Set<String> configSet) {
        if (CollectionUtils.isEmpty(userCrowdSet) || CollectionUtils.isEmpty(configSet)) {
            return false;
        }
        for (String id : configSet) {
            if (userCrowdSet.contains(id)) {
                return true;
            }
        }
        return false;
    }
}
