package com.bxm.adx.common.market.exchange.rebuild.offer;

import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.buy.dispatcher.Dispatcher;
import com.bxm.adx.common.buy.dispatcher.DispatcherPriceConfig;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.response.Bid;
import org.springframework.context.annotation.Configuration;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.*;

/**
 * @author fgf
 * @date 2023/1/10
 **/
@Configuration
public class NoneDspAvgPriceOffer implements Offer {
    @Override
    public Map<Bid, OfferResult> offer(Collection<Bid> bids, BidRequest request, Dispatcher dispatcher) {
        Map<Bid, OfferResult> offerResultMap = new HashMap<>(bids.size());
        for (Bid bid : bids) {
            BigDecimal offer = getPriceByConfig(request, dispatcher, bid.getPrice());
            offerResultMap.put(bid, OfferResult.builder()
                            .offer(offer)
                            .chargeType(AdxConstants.ChargeType.CPM)
                    .build());
        }
        return offerResultMap;
    }

    @Override
    public OfferType offerType() {
        return OfferType.NONE_Dsp_Avg_Price;
    }

    /**
     * 获取配置出价配置的情况下的出价
     * @param bidRequest
     * @param dispatcher
     * @param dspOffer 单位分
     * @return
     */
    private BigDecimal getPriceByConfig(BidRequest bidRequest, Dispatcher dispatcher, BigDecimal dspOffer) {
        DispatcherPriceConfig config = dispatcher.getDispatcherPriceConfig();
        if (Objects.isNull(config)) {
            return dspOffer;
        }
        //媒体出价=dsp出价*（1-利润率）*出价系数 注:此处dsp出价是经过流量分配中广告主系数处理后的出价，非原始dsp出价
        BigDecimal newDspOffer = getDspPriceByDispatcher(dspOffer, dispatcher);
        BigDecimal profitMargin = Optional.ofNullable(config.getProfitMargin()).orElse(BigDecimal.ZERO);
        BigDecimal newPrice = newDspOffer
                .multiply(new BigDecimal(100.00).subtract(profitMargin))
                .multiply(config.getBiddingCoefficient()).movePointLeft(2);
        BigDecimal maxBidFloor = maxBidFloor(bidRequest, dispatcher);
        if (newPrice.compareTo(maxBidFloor) >= 0) {
            return newPrice.setScale(0, RoundingMode.HALF_UP);
        } else {
            //探索出价-底价加1分钱
            BigDecimal explorePrice = maxBidFloor.add(BigDecimal.ONE);
            if (newDspOffer.compareTo(explorePrice) >= 0) {
                return explorePrice.setScale(0, RoundingMode.HALF_UP);
            }
        }

        return null;
    }

    /**
     * 取配置底价和请求底价中较大的那个
     * @param bidRequest
     * @param dispatcher
     * @return
     */
    private BigDecimal maxBidFloor(BidRequest bidRequest, Dispatcher dispatcher) {
        //元转分
        BigDecimal dispatcherBidFloor = Optional.ofNullable(dispatcher.getDspBasePrice()).orElse(BigDecimal.ZERO).movePointRight(2);
        BigDecimal bidFloor = bidRequest.getImps().iterator().next().getBid_floor();
        int result = bidFloor.compareTo(dispatcherBidFloor);
        switch (result) {
            case -1:
                return dispatcherBidFloor;
            case 0:
            case 1:
            default:
                return bidFloor;
        }
    }

    /**
     * 计算广告主系数处理后的dsp出价
     * 例子：广告主出价10元，8元以内部分折扣系数0.8，8元以外部分0.4，则上报广告主赢价=8*0.8+（10-8）*0.4
     * @param dspOffer
     * @param dispatcher
     * @return
     */
    private static BigDecimal getDspPriceByDispatcher(BigDecimal dspOffer, Dispatcher dispatcher) {
        DispatcherPriceConfig config = dispatcher.getDispatcherPriceConfig();
        //未配置出价配置直接返回原出价
        if (Objects.isNull(config)) {
            return dspOffer;
        }
        BigDecimal priceSplit = config.getPriceSplit();
        BigDecimal lt = config.getLtPriceDiscountFactor();
        BigDecimal gt = config.getGtPriceDiscountFactor();
        //未配置的时段使用默认系数
        if (Objects.isNull(priceSplit)) {
            lt = Optional.ofNullable(lt).orElse(BigDecimal.ONE);
            return dspOffer.multiply(lt);
        } else {
            priceSplit = priceSplit.movePointRight(2);
            //看注释例子
            int compare = dspOffer.compareTo(priceSplit);
            if (compare <= 0) {
                return dspOffer.multiply(lt);
            } else {
                BigDecimal ltPrice = priceSplit.multiply(lt);
                BigDecimal gtPrice = dspOffer.subtract(priceSplit).multiply(gt);
                return ltPrice.add(gtPrice);
            }
        }
    }
}
