package com.bxm.adx.common.market.filter;

import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.market.Deal;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.BidResponse;
import com.bxm.adx.common.sell.response.Bid;
import com.bxm.adx.common.sell.response.SeatBid;
import com.bxm.mcssp.common.enums.app.DockingMethodTypeEnum;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.context.annotation.Configuration;

import java.math.BigDecimal;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @author fgf
 * @date 2023/3/21
 **/
@Slf4j
@Configuration
@DealFilterCondition(preview = true)
public class PriceBidFloorFilter implements Filter {
    @Override
    public void filter(List<Deal> deals, Set<Deal> trash) {
        Set<Deal> remove = deals.stream().filter(
                deal -> {
                    BidResponse response = deal.getBidResponse();
                    BidRequest request = deal.getBidRequest();
                    List<SeatBid> seatBids = response.getSeat_bid();
                    seatBids.removeIf(seatBid -> {
                        List<Bid> bidList = seatBid.getBid();
                        bidList.removeIf(bid -> {
                            BigDecimal price = bid.getPrice();
                            if (null == price) {
                                return true;
                            }
                            //cpc模式下暂不过滤,后续还需改动
                            if (isCpc(request)) {
                                return false;
                            }
                            //api接入模式下需要进行底价过滤
                            BigDecimal bidFloor = null;
                            if (isApi(request)) {
                                bidFloor = request.getImps().iterator().next().getBid_floor();
                            }
                            if (Objects.nonNull(bidFloor)) {
                                if (price.compareTo(bidFloor) < 0) {
                                    return true;
                                }
                            }
                            return false;
                        });
                        return CollectionUtils.isEmpty(bidList);
                    });
                    return CollectionUtils.isEmpty(seatBids);
                }
        ).collect(Collectors.toSet());
        if (CollectionUtils.isNotEmpty(remove)) {
            trash.addAll(remove);
        }
    }

    @Override
    public int getOrder() {
        return 1;
    }

    private boolean isApi(BidRequest request) {
        if (Objects.nonNull(request.getDockingMethodType())
                && DockingMethodTypeEnum.API_OPERATION.getType().equals(request.getDockingMethodType())) {
            return true;
        }
        return false;
    }

    private boolean isCpc(BidRequest request) {
        int bidModel = Optional.ofNullable(request.getBid_model()).orElse(AdxConstants.BidModel.SUPPORT_CPM);
        if (AdxConstants.BidModel.SUPPORT_CPC == bidModel) {
            return true;
        }
        return false;
    }
}
