package com.bxm.adx.common.sell.position.mapping;

import com.bxm.adx.common.AdxProperties;
import com.bxm.adx.common.PositionMappingProperties;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.position.Position;
import com.bxm.adx.common.sell.position.PositionChangeHandler;
import com.bxm.adx.common.sell.request.Impression;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @author fgf
 * @date 2023/2/22
 **/
@Component
public class PositionTypeStrategy implements PositionMappingStrategy, PositionChangeHandler {
    private final AdxProperties adxProperties;
    /**
     * 广告位尺寸类型-广告位集合
     * <pre>
     * |----------------------------|-----------------------------|
     * |    Key                     | Value                       |
     * |----------------------------|-----------------------------|
     * | {appId-positionScene}      |[position,position,...]      |
     * |----------------------------|-----------------------------|
     * </pre>
     */
    private final ConcurrentHashMap<String, Position> positionTypeMap = new ConcurrentHashMap<>();

    public PositionTypeStrategy(AdxProperties adxProperties) {
        this.adxProperties = adxProperties;
    }

    @Override
    public Strategy strategy() {
        return Strategy.Position_type;
    }

    @Override
    public Position mapping(BidRequest bidRequest, Impression impression, PositionMappingProperties properties) {
        String appId = getAppId(bidRequest, properties);
        return getPositionByInfo(appId, impression.getImp_type());
    }

    /**
     * 获取复杂广告位映射（比如bes）
     *
     * @param appId
     * @param positionScene
     * @return
     */
    public Position getPositionByInfo(String appId, Integer positionScene) {
        if (StringUtils.isEmpty(appId)) {
            return null;
        }
        if (Objects.isNull(positionScene)) {
            return null;
        }
        String key = buildMapKeyByPositionType(appId, positionScene);
        return positionTypeMap.get(key);
    }

    /**
     * 根据请求的os区别获取bes对应的appId
     *
     * @param bidRequest
     * @return
     */
    private String getAppId(BidRequest bidRequest, PositionMappingProperties properties) {
        return properties.getDefaultAppId();
    }

    @Override
    public void doUpdate(Position old, Position latest) {
        if (isExecuteUpdateAndDoDelete(old, latest)) {
            Strategy strategy = getStrategy(latest);
            switch (strategy) {
                case Position_type:
                    String infoKey = buildMapKeyByPositionType(latest);
                    if (StringUtils.isNotEmpty(infoKey)) {
                        positionTypeMap.put(infoKey, latest);
                    }
                    break;
            }
        }
    }

    @Override
    public void doDelete(Position old) {

        Strategy strategy = getStrategy(old);
        switch (strategy) {
            case Position_type:
                String infoKey = buildMapKeyByPositionType(old);
                if (StringUtils.isNotEmpty(infoKey)) {
                    positionTypeMap.remove(infoKey);
                }
                break;
        }
    }

    /**
     * 构建复杂广告位映射key
     *
     * @param position
     * @return
     */
    public String buildMapKeyByPositionType(Position position) {
        String appId = position.getAppId();
        Byte scene = position.getPositionScene();
        if (StringUtils.isEmpty(appId)) {
            return null;
        }
        if (Objects.isNull(scene)) {
            return null;
        }
        String val = new StringBuilder(appId)
                .append("-")
                .append(position.getPositionScene())
                .toString();
        return val;
    }

    /**
     * 构建复杂广告位映射key
     *
     * @param appId
     * @param positionScene
     * @return
     */
    public String buildMapKeyByPositionType(String appId, int positionScene) {
        String val = new StringBuilder(appId)
                .append("-")
                .append(positionScene)
                .toString();
        return val;
    }

    private Strategy getStrategy(Position position) {
        String appId = position.getAppId();
        List<PositionMappingProperties> positionMapping = adxProperties.getPositionMapping();
        if (CollectionUtils.isEmpty(positionMapping)) {
            return Strategy.Default;
        }
        for (PositionMappingProperties pm : positionMapping) {
            if (appId.equals(pm.getAndroidAppId())
                    || appId.equals(pm.getIosAppId())
                    || appId.equals(pm.getDefaultAppId())) {
                return pm.getStrategy();
            }
        }
        return Strategy.Default;
    }
}
