package com.bxm.adx.common.utils;

import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.FileUtils;
import org.springframework.beans.factory.InitializingBean;

import java.io.File;
import java.util.concurrent.RejectedExecutionHandler;
import java.util.function.Consumer;

/**
 * 写入文件的异步处理。
 *
 * @author allen
 * @date 2021-07-19
 * @since 1.0
 */
@Slf4j
public abstract class AbstractWriteToFileSyncProcessor extends AbstractSyncProcessor<AbstractWriteToFileSyncProcessor.Line> implements InitializingBean {

    private final String parentDir;

    public AbstractWriteToFileSyncProcessor(int coreSize, int maximumLength, String threadName,
                                            RejectedExecutionHandler rejectedExecutionHandler, String parentDir) {
        super(coreSize, maximumLength, threadName, rejectedExecutionHandler);
        this.parentDir = parentDir;
    }

    @Override
    protected Consumer<Line> getConsumer() {
        return new Consumer<Line>() {
            @Override
            public void accept(Line line) {
                try {
                    String fileName = line.getFileName();
                    File file = new File(parentDir + fileName);
                    FileUtils.writeByteArrayToFile(file, line.getData(), line.isAppend());
                } catch (Exception e) {
                    log.error("writeLines: ", e);
                }
            }
        };
    }

    @Override
    public void afterPropertiesSet() {
        FileHelper.createDirIfNecessary(parentDir);
    }

    @Data
    @Builder
    @AllArgsConstructor
    public static class Line {

        /**
         * 文件名
         */
        private String fileName;
        /**
         * 写入数据
         */
        private byte[] data;
        /**
         * 如果为true ，则字节将添加到文件末尾而不是覆盖
         */
        private boolean append;
    }
}
