package com.bxm.adx.common.buy.dispatcher;

import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.adapter.AdxContextFactory;
import com.bxm.adx.common.buy.dispatcher.abtest.DispatcherABConfig;
import com.bxm.adx.common.buy.dispatcher.abtest.DispatcherABConfigCaching;
import com.bxm.adx.common.buy.dispatcher.abtest.cached.DispatcherCached;
import com.bxm.adx.common.buy.dispatcher.abtest.cached.DispatcherConfigCached;
import com.bxm.adx.common.buy.dispatcher.filter.DispatcherFilterFactory;
import com.bxm.adx.common.buy.dispatcher.reorder.DispatcherReorderFactory;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.position.Position;
import com.bxm.adx.common.utils.AdxUtils;
import com.bxm.adx.common.utils.DateUtils;
import com.bxm.adx.common.utils.MapHelper;
import com.bxm.adx.facade.constant.enums.AdxErrEnum;
import com.bxm.adx.facade.exception.AdxException;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.RandomStringUtils;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.StringUtils;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.time.temporal.ChronoField;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author allen
 * @since 2019-12-16
 */
@Configuration
@Slf4j
public class DispatcherServiceImpl implements DispatcherService {

    private final static Integer DELAY = 15;

    private final DispatcherConfigCached dispatcherConfigCached;
    private final DispatcherABConfigCaching dispatcherABConfigCaching;
    private final DispatcherCached dispatcherCached;
    private final DispatcherFilterFactory dispatcherFilterFactory;
    private final DispatcherReorderFactory dispatcherReorderFactory;

    public DispatcherServiceImpl(DispatcherConfigCached dispatcherConfigCached,
                                 DispatcherABConfigCaching dispatcherABConfigCaching, DispatcherCached dispatcherCached,
                                 DispatcherFilterFactory dispatcherFilterFactory, DispatcherReorderFactory dispatcherReorderFactory) {
        this.dispatcherConfigCached = dispatcherConfigCached;
        this.dispatcherABConfigCaching = dispatcherABConfigCaching;
        this.dispatcherCached = dispatcherCached;
        this.dispatcherFilterFactory = dispatcherFilterFactory;
        this.dispatcherReorderFactory = dispatcherReorderFactory;
    }

    @Override
    public Map<Integer, Collection<Dispatcher>> getPriority(Position position, BidRequest bidRequest) {
        Map<Integer, Collection<Dispatcher>> dsps = Maps.newTreeMap();
        Collection<Dispatcher> dispatchers = getDispatchers(position);
        dispatchers = dispatchers.stream()
                .filter(dispatcherDspCacheVO -> dispatcherDspCacheVO.getOpened() == Dispatcher.DISPATCHER_OPENED_YES)
                .collect(Collectors.toSet());
        DispatcherContext<Dispatcher> context = DispatcherContext.<Dispatcher>builder()
                .values(dispatchers)
                .position(position)
                .request(bidRequest)
                .build();
        //过滤
        dispatcherFilterFactory.filter(context);
        //排序
        dispatchers = dispatcherReorderFactory.reorder(context);

        for (Dispatcher dispatcher : dispatchers) {
            int priority = dispatcher.getPriority();
            Collection<Dispatcher> collection = MapHelper.get(dsps, priority, new ArrayList<>());
            //设置价格相关参数
            dispatcher.setDispatcherPriceConfigs(getDispatcherPriceConfig(dispatcher));
            collection.add(dispatcher);
        }
        return dsps;
    }

    /**
     * 获取广告位流量分配信息
     *
     * @param position
     * @return
     */
    private Collection<Dispatcher> getDispatchers(Position position) {
        String positionId = position.getPositionId();
        DispatcherABConfig dispatcherABConfig = dispatcherABConfigCaching.get(positionId);
        if (Objects.isNull(dispatcherABConfig)) {
            return Sets.newHashSet();
        }

        //先进行用户分桶
        String uid = AdxContextFactory.get().getUid();
        if (StringUtils.isEmpty(uid)) {
            uid = RandomStringUtils.randomAlphabetic(13);
        }
        int userBucket = AdxUtils.getBucket(uid);
        //根据分桶拿到对应的流量分配配置id
        Long configId = dispatcherConfigCached.getConfigId(positionId, userBucket);
        //如果分桶失败，则使用旧分配数据
        if (Objects.isNull(configId)) {
            if (log.isWarnEnabled()) {
                log.warn("position {} user bucket {} fail ", positionId, userBucket);
            }
            return Sets.newHashSet();

        }
        AdxContextFactory.get().setConfigId(configId);
        return dispatcherCached.getDispatchers(positionId, configId);
    }

    /**
     * 获取价格相关设置
     * @param dispatcher
     */
    private List<DispatcherPriceConfig> getDispatcherPriceConfig(Dispatcher dispatcher) {
        List<DispatcherPriceConfig> priceConfigs = getPriceConfigsByTime(dispatcher);
        if (Objects.isNull(priceConfigs)) {
            return null;
        }
        if (CollectionUtils.isEmpty(priceConfigs)) {
            log.warn("position {} dispatcher {} price config empty err", dispatcher.getPositionId(), dispatcher.getId());

            if (AdxConstants.DispatcherAvgType.FIXED_DISCOUNT == dispatcher.getAvgType()) {
                log.error("position {} dispatcher {} avg-type {} err", dispatcher.getPositionId(), dispatcher.getId(), dispatcher.getAvgType());
                throw new AdxException(AdxErrEnum.DISPATCHER_ERR);
            }
            //默认配置，在合适的时机废除或者调整该配置，默认配置可能会导致一些意外的很难被感知的情况
            priceConfigs.add(DispatcherPriceConfig.builder()
                    .ltPriceDiscountFactor(BigDecimal.ONE)
                    .gtPriceDiscountFactor(BigDecimal.ONE)
                    .biddingCoefficient(BigDecimal.ONE)
                    .profitMargin(BigDecimal.ZERO)
                    .build());
        }
        return priceConfigs;
    }

    /**
     * 获取价格分时段设置
     * @return 所有策略里配置的时段
     */
    private List<DispatcherPriceConfig> getPriceConfigsByTime(Dispatcher dispatcher) {
        List<DispatcherPriceConfig> configs = dispatcher.getDispatcherDspPosPriceConfigVOS();
        if (CollectionUtils.isEmpty(configs)) {
            return null;
        }
        List<DispatcherPriceConfig> priceConfigs = new ArrayList<>();
        int hour = LocalDateTime.now().get(ChronoField.HOUR_OF_DAY);
        for (DispatcherPriceConfig config : configs) {
            Integer opened = config.getOpened();
            if (opened != null && opened == AdxConstants.YES) {
                continue;
            }
            Integer start = config.getStartTime();
            Integer end = config.getEndTime();
            if (Objects.nonNull(start) && Objects.nonNull(end)) {
                if (start <= hour && hour < end) {
                    bidOptimizerConfig(config, dispatcher);
                    priceConfigs.add(config);
                }
            }
        }
        return priceConfigs;
    }

    /**
     * 介入价格策略
     * @param config
     * @param dispatcher
     */
    private void bidOptimizerConfig(DispatcherPriceConfig config, Dispatcher dispatcher) {
        if (Objects.nonNull(dispatcher.getCpcControl()) && dispatcher.getCpcControl() == Dispatcher.DISPATCHER_OPENED_YES) {
            Date date = dispatcher.getProjectTime();
            long duration = DateUtils.getDuration(date);
            if (duration <= DELAY) {
                BigDecimal coefficient = dispatcher.getProjectBiddingCoefficient();
                BigDecimal discount = dispatcher.getProjectDiscount();
                if (Objects.nonNull(coefficient) && Objects.nonNull(discount)) {
                    config.setBiddingCoefficient(coefficient);
                    config.setGtPriceDiscountFactor(discount);
                    config.setLtPriceDiscountFactor(discount);
                }
            }
        }
    }
}
