package com.bxm.adx.common.buy.dispatcher.reorder;

import com.bxm.adx.common.CacheKeys;
import com.bxm.adx.common.adapter.AdxContextFactory;
import com.bxm.adx.common.buy.dispatcher.Dispatcher;
import com.bxm.adx.common.buy.dispatcher.DispatcherContext;
import com.bxm.adx.facade.constant.redis.AdxKeyGenerator;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.context.annotation.Configuration;

import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author fgf
 * @date 2023/3/13
 **/
@Slf4j
@Configuration
public class DispatcherSdkReorder implements DispatcherReorder {
    private final Fetcher fetcher;
    private final Updater updater;

    public DispatcherSdkReorder(Fetcher fetcher, Updater updater) {
        this.fetcher = fetcher;
        this.updater = updater;
    }

    @Override
    public Collection<Dispatcher> reorder(DispatcherContext<Dispatcher> context) {
        Collection<Dispatcher> dispatchers = context.getValues();
        //是否配置频次
        Optional exist = dispatchers.stream().filter(dispatcher -> Objects.nonNull(dispatcher.getFrequency())).findFirst();
        if (!exist.isPresent()) {
            return dispatchers;
        }
        Collection<Dispatcher> reorderList = null;
        String positionId = context.getPosition().getPositionId();
        String uid = AdxContextFactory.get().getUid();
        //获取上一次排序结果
        Collection<Dispatcher> recordList = fetcher.hfetchList(CacheKeys.Dispather.getDispatcherKeyGeneratorByUid(positionId), uid, Dispatcher.class);
        reorderList = CollectionUtils.isEmpty(recordList) ? context.getValues() : recordList;
        //根据最新频次状况重排序
        reorderList = reorderPriorityByFreq(reorderList, uid, positionId);
        return reorderList;
    }

    @Override
    public DispatcherOrderStrategy orderStrategy() {
        return DispatcherOrderStrategy.sdk_common;
    }


    /**
     * 根据频次重排序
     *
     * @param ds
     * @param uid
     * @return
     */
    private Collection<Dispatcher> reorderPriorityByFreq(Collection<Dispatcher> ds, String uid, String positionId) {
        Set<Dispatcher> result = Sets.newHashSet();
        //获取排好序的优先级
        List<Integer> ps = ds.stream().map(Dispatcher::getPriority).sorted().distinct().collect(Collectors.toList());
        //获取最大优先级max，达到频次的dsp的优先级调整为max+1
        Integer max = ps.get(ps.size() - 1);
        for (Integer p : ps) {
            //获取当前优先级状态正常的dsp
            Set<Dispatcher> dispatchers = ds.stream()
                    .filter(dispatcherDspCacheVO -> dispatcherDspCacheVO.getPriority() == p)
                    .collect(Collectors.toSet());
            //当同优先级有多个dsp时频次重排序不起效，不调整优先级
            if (dispatchers.size() == 1) {
                Dispatcher dispatcher = dispatchers.iterator().next();
                //设置的频次
                Integer initFreq = dispatcher.getFrequency();
                //频次为null则视为无限频次，无限频次不需要重排序
                if (null != initFreq) {
                    //已用频次
                    Integer used = getPositionDspFreq(dispatcher, uid);
                    //频次已经用完
                    if (used >= initFreq) {
                        //如果当前分配优先级就是最大级则不用重置优先级
                        if (dispatcher.getPriority() != max.intValue()) {
                            max = max + 1;
                            //调整dsp优先级，并调整当前最大优先级
                            dispatcher.setPriority(max);
                        }
                        resetPositionDspFreq(dispatcher, uid);
                    }
                }
                result.add(dispatcher);
            } else {
                result.addAll(dispatchers);
            }
        }
        //记录重排序结果给下一次用，当天24点过期
        updateReorderDispatcher(uid, positionId, result);
        return result;
    }

    /**
     * 获取已用频次
     *
     * @param dispatcher
     * @param uid
     * @return
     */
    private int getPositionDspFreq(Dispatcher dispatcher, String uid) {
        KeyGenerator keyGenerator = getFreqKeyGenerator(dispatcher, uid);
        Long freq = fetcher.fetch(keyGenerator, Long.class);
        return Objects.isNull(freq) ? 0 : freq.intValue();
    }

    /**
     * 重置已用频次
     *
     * @param dispatcher
     * @param uid
     */
    private void resetPositionDspFreq(Dispatcher dispatcher, String uid) {
        LocalDateTime midnight = LocalDateTime.now().plusDays(1).withHour(0).withMinute(0).withSecond(0).withNano(0);
        long seconds = ChronoUnit.SECONDS.between(LocalDateTime.now(), midnight);
        KeyGenerator keyGenerator = getFreqKeyGenerator(dispatcher, uid);
        updater.update(keyGenerator, 0, (int) seconds);
    }

    /**
     * 缓存重排序
     *
     * @param uid
     * @param positionId
     * @param dispatchers
     */
    private void updateReorderDispatcher(String uid, String positionId, Collection<Dispatcher> dispatchers) {
        LocalDateTime midnight = LocalDateTime.now().plusDays(1).withHour(0).withMinute(0).withSecond(0).withNano(0);
        long seconds = ChronoUnit.SECONDS.between(LocalDateTime.now(), midnight);
        updater.hupdate(CacheKeys.Dispather.getDispatcherKeyGeneratorByUid(positionId), uid, dispatchers, (int) seconds);
    }

    /**
     * 获取频次rediskey
     *
     * @param dispatcher
     * @param uid
     * @return
     */
    private KeyGenerator getFreqKeyGenerator(Dispatcher dispatcher, String uid) {
        return AdxKeyGenerator.Counter.getPositionDspFrequency(dispatcher.getPositionId(),
                dispatcher.getConfigId().toString(), dispatcher.getDspId().toString(), uid);
    }
}
