package com.bxm.adx.common.limiter;

import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.utils.NamedThreadFactory;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.DisposableBean;

import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * @author allen
 * @date 2022-04-26
 * @since 1.0
 */
@Slf4j
public class DefaultDeleteExpiredHandler implements DeleteExpiredHandler, DisposableBean {

    private final BlockingQueue<QueueEntity> queue = new LinkedBlockingQueue<>(1000);
    private final ThreadPoolExecutor executor = new ThreadPoolExecutor(1, 1, 0, TimeUnit.MILLISECONDS,
            new LinkedBlockingQueue<>(), new NamedThreadFactory("expire"));
    private final Updater updater;

    public DefaultDeleteExpiredHandler(Updater updater) {
        this.updater = updater;
        this.startHandleQueue();
    }

    @Override
    public void delete(KeyGenerator hashKey, String field) {
        // 如果队列满了，直接丢弃了。
        if (!queue.offer(new QueueEntity(hashKey, field))) {
            if (log.isDebugEnabled()) {
                log.debug("the queue was full.");
            }
        }
    }

    @Override
    public void destroy() {
        this.executor.shutdownNow();
    }

    private void startHandleQueue() {
        executor.execute(() -> {
            for (;;) {
                try {
                    QueueEntity entity = queue.take();
                    KeyGenerator hashKey = entity.hashKey;
                    String field = entity.field;
                    updater.hremove(hashKey, field);
                    if (log.isDebugEnabled()) {
                        log.debug("Deleted key hash {} field {}.", hashKey.generateKey(), field);
                    }
                } catch (Exception e) {
                    log.info("take: {}", e.getMessage());
                }
            }
        });
    }

    private static class QueueEntity {

        KeyGenerator hashKey;
        String field;

        private QueueEntity(KeyGenerator hashKey, String field) {
            this.hashKey = hashKey;
            this.field = field;
        }
    }
}
