package com.bxm.adx.common.log.dsplog;


import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.adapter.BidModelAdapter;
import com.bxm.adx.common.buy.dispatcher.Dispatcher;
import com.bxm.adx.common.log.AdxDataLog;
import com.bxm.adx.common.log.AdxDataLogMonitor;
import com.bxm.adx.common.log.buriedSwitch.BuriedSwitchDao;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.BidResponse;
import com.bxm.adx.common.sell.response.SeatBid;
import com.bxm.warcar.integration.distributed.DisableDistributed;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.NamedThreadFactory;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * 记录dsp的请求和响应 分开文件进行记录
 *
 * @author zhangdong
 * @date 2022/11/30
 */
@Component
@Slf4j
public class DspLogRecord implements DisposableBean{

    private final ObjectMapper mapper = new ObjectMapper();

    private final Logger requestLogger = LoggerFactory.getLogger(DspLogRecord.class.getName() + ".request");

    private final Logger responseLogger = LoggerFactory.getLogger(DspLogRecord.class.getName() + ".response");

    /**
     * 广告主原始响应数据
     */
    private final Logger dspOriginalResponseLogger = LoggerFactory.getLogger(DspLogRecord.class.getName() + ".originalResponse");
    /**
     * 广告主原始响应转换为bxm协议之后的响应
     */
    private final Logger bxmOriginalResponseLogger = LoggerFactory.getLogger(DspLogRecord.class.getName() + ".bxmOriginalResponse");

    /**
     * bxm参竟返回
     */
    private final Logger bxmResponseLogger = LoggerFactory.getLogger(DspLogRecord.class.getName() + ".bxmResponse");
    /**
     * 媒体参竟返回
     */
    private final Logger mediaResponseLogger = LoggerFactory.getLogger(DspLogRecord.class.getName() + ".mediaResponse");

    private final ThreadPoolExecutor executor = new ThreadPoolExecutor(10, 20, 0, TimeUnit.MILLISECONDS,
            new LinkedBlockingQueue<>(1000), new NamedThreadFactory("dspLogResponse"), new ThreadPoolExecutor.CallerRunsPolicy());

    private final ThreadPoolExecutor buriedExecutor = new ThreadPoolExecutor(10, 20, 0, TimeUnit.MILLISECONDS,
            new LinkedBlockingQueue<>(1000), new NamedThreadFactory("dspBuriedResponse"), new ThreadPoolExecutor.DiscardPolicy());

    private final ConcurrentHashMap<String, Long> map = new ConcurrentHashMap<>();

    private final BuriedSwitchDao buriedSwitchDao;

    private final AdxDataLogMonitor adxDataLogMonitor;

    public DspLogRecord(BuriedSwitchDao buriedSwitchDao, AdxDataLogMonitor adxDataLogMonitor) {
        this.buriedSwitchDao = buriedSwitchDao;
        this.adxDataLogMonitor = adxDataLogMonitor;
    }

    @PostConstruct
    public void init() {
        mapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
    }

    public void responseLog(BidResponse bidResponse,byte[] mediaResponse, BidModelAdapter adapter) {
        if (bidResponse == null || CollectionUtils.isEmpty(bidResponse.getSeat_bid())) {
            return;
        }
        for (SeatBid sb : bidResponse.getSeat_bid()) {
            if (buriedSwitchDao.getBuriedSwitch(sb.getPositionId(), sb.getDspId(), sb.getDspPosid())) {
                AdxDataLog mediaResponseLog = new AdxDataLog(bidResponse.getId(), sb.getPositionId(), sb.getDspId(), sb.getDspPosid(),sb.getDspAppid(),adapter.getRequestFormat(),null,adapter.convertSspResponse(mediaResponse), null);
                AdxDataLog bxmResponseLog = new AdxDataLog(bidResponse.getId(), sb.getPositionId(), sb.getDspId(), sb.getDspPosid(),sb.getDspAppid(),AdxConstants.Format.Json,null,JsonHelper.convert(bidResponse), null);

                try {
                    log(mapper.writeValueAsString(bxmResponseLog),bxmResponseLogger);
                    log(mapper.writeValueAsString(mediaResponseLog),mediaResponseLogger);
                    adxDataLogMonitor.responseMonitor(sb.getPositionId(),sb.getDspId(),sb.getDspPosid());
                } catch (Exception e) {
                    log.error("Jackson JSON 解析错误", e);
                }
            }
        }
    }

    /**
     *  @param request
     * @param str dsp原始响应
     * @param format dsp协议类型
     * @param response dsp响应转成的bxmResponse
     * @param dispatcher
     */
    public void dspLog(BidRequest request, String str, AdxConstants.Format format, BidResponse response, Dispatcher dispatcher) {
        try {
            logRequest(request);
            if (buriedSwitchDao.getBuriedSwitch(dispatcher.getPositionId(),dispatcher.getDspId(),dispatcher.getDspPosid())) {
                logDspOriginalResponse(new AdxDataLog(request.getId(),dispatcher.getPositionId(), dispatcher.getDspId(), dispatcher.getDspPosid(),dispatcher.getDspAppid(),format,null,str, null),dspOriginalResponseLogger);
                logDspOriginalResponse(new AdxDataLog(request.getId(),dispatcher.getPositionId(), dispatcher.getDspId(), dispatcher.getDspPosid(),dispatcher.getDspAppid(),AdxConstants.Format.Json,null, JsonHelper.convert(response), null),bxmOriginalResponseLogger);
                adxDataLogMonitor.originalResponseMonitor(dispatcher.getPositionId(),dispatcher.getDspId(),dispatcher.getDspPosid());
            }
            //填充数据
            if (StringUtils.isEmpty(response.getId())) {
                response.setId(request.getId());
            }
            for (SeatBid seatBid : response.getSeat_bid()) {
                seatBid.setDspId(dispatcher.getDspId());
            }
            //盘古的数据不存
            if (dispatcher.getDspId().intValue() == 105) {
                return;
            }
            logResponse(response);
        } catch (Exception e) {}
    }

    private void logDspOriginalResponse(AdxDataLog dataLog,Logger logger) {
        if (dataLog == null) {
            return;
        }
        try {
            log(mapper.writeValueAsString(dataLog),logger);
        } catch (Exception e) {
            log.error("logOriginalResponse error", e);
        }
    }

    private void log(String s,Logger logger) {
        if (StringUtils.isNotBlank(s)) {
            buriedExecutor.execute(() -> logger.info(s));
        }
    }

    private void logResponse(BidResponse obj) {
        if (obj == null) {
            return;
        }
        try {
            String s = mapper.writeValueAsString(obj);
            executor.execute(() -> responseLogger.info(s));
        } catch (Exception e) {
            log.error("jackson json error", e);
        }

    }


    private void logRequest(BidRequest obj) {
        if (obj == null) {
            return;
        }
        if (map.containsKey(obj.getId())) {
            return;
        }
        map.put(obj.getId(), System.currentTimeMillis());
        try {
            String s = mapper.writeValueAsString(obj);
            executor.execute(() -> requestLogger.info(s));
        } catch (Exception e) {
            log.error("jackson json error", e);
        }

    }

    @DisableDistributed
    @Scheduled(cron = "*/1 * * * * ?")
    public void deleteExpireKey() {
        long l = System.currentTimeMillis();
        for (Map.Entry<String, Long> entry : map.entrySet()) {
            if ((l - entry.getValue()) > 1000) {
                map.remove(entry.getKey());
            }
        }
    }


    @Override
    public void destroy() {
        executor.shutdown();
        try {
            executor.awaitTermination(5, TimeUnit.SECONDS);
        } catch (Exception ignored) {

        }
    }
}
