package com.bxm.adx.common.market.exchange;

import com.bxm.adx.common.buy.Buyer;
import com.bxm.adx.common.buy.dispatcher.Dispatcher;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.warcar.ip.IP;
import com.google.common.collect.Maps;

import java.util.Map;

/**
 * 买方交易上下文，线程独享。
 * 可以通过 {@link #put(Object, Object)} 来设置一些内容，然后再使用 {@link #get(Object)} 来获取。
 * @author allen
 * @date 2020-11-05
 * @since 1.0
 */
public class ExchangeContext {

    private static final ThreadLocal<ExchangeContext> THREAD_LOCAL = new ThreadLocal<ExchangeContext>() {
        @Override
        protected ExchangeContext initialValue() {
            return new ExchangeContext();
        }
    };

    private static final String KEY_BIDREQUEST = "$$BID_REQUEST$$";

    private static final String KEY_ADXREQUEST = "$$ADX_REQUEST$$";
    private static final String KEY_BUYER = "$$BUYER$$";
    @Deprecated
    private static final String KEY_DISPATCH_CONFIG = "$$DISPATCH_CONFIG$$";
    private static final String KEY_DISPATCH = "$$DISPATCH$$";
    private static final String KEY_PLUGINLOG = "$$PLUGINLOG$$";
    private static final String KEY_IP = "$$IP$$";

    private final Map<Object, Object> attachment = Maps.newHashMap();

    private static ExchangeContext get() {
        return THREAD_LOCAL.get();
    }

    public static void remove() {
        THREAD_LOCAL.remove();
    }

    static void putBidRequest(BidRequest bidRequest) {
        get().attachment.put(KEY_BIDREQUEST, bidRequest);
    }

    static void putAdxRequest(BidRequest adxRequest) {
        get().attachment.put(KEY_ADXREQUEST, adxRequest);
    }

    static void putBuyer(Buyer buyer) {
        get().attachment.put(KEY_BUYER, buyer);
    }

    @Deprecated
    static void putDispatchConfig(Long configId) {
        get().attachment.put(KEY_DISPATCH_CONFIG, configId);
    }

    static void putDispatch(Dispatcher dispatcher) {
        get().attachment.put(KEY_DISPATCH, dispatcher);
    }

    static void putDataLog(boolean log) {
        get().attachment.put(KEY_PLUGINLOG, log);
    }

    static void putIp(IP ip) {
        get().attachment.put(KEY_IP, ip);
    }

    public static void put(Object key, Object value) {
        ExchangeContext context = get();
        context.attachment.put(key, value);
    }

    public static Object get(Object key) {
        ExchangeContext context = get();
        return context.attachment.get(key);
    }

    public static BidRequest getBidRequest() {
        Object o = get().attachment.get(KEY_BIDREQUEST);
        if (o instanceof BidRequest) {
            return (BidRequest) o;
        }
        return null;
    }

    public static BidRequest getAdxRequest() {
        Object o = get().attachment.get(KEY_ADXREQUEST);
        if (o instanceof BidRequest) {
            return (BidRequest) o;
        }
        return null;
    }

    public static Buyer getBuyer() {
        Object o = get().attachment.get(KEY_BUYER);
        if (o instanceof Buyer) {
            return (Buyer) o;
        }
        return null;
    }
    @Deprecated
    public static Long getDispatcherConfigId() {
        Object o = get().attachment.get(KEY_DISPATCH_CONFIG);
        if (o instanceof Long) {
            return (Long) o;
        }
        return null;
    }

    public static Dispatcher getDispatcher() {
        Object o = get().attachment.get(KEY_DISPATCH);
        if (o instanceof Dispatcher) {
            return (Dispatcher) o;
        }
        return null;
    }

    public static boolean getPluginLog() {
        Object o = get().attachment.get(KEY_PLUGINLOG);
        if (!(o instanceof Boolean)) {
            return false;
        }
        return (boolean) o;
    }

    public static IP getIp() {
        Object o = get().attachment.get(KEY_IP);
        if (o instanceof IP) {
            return (IP) o;
        }
        return null;
    }
}
