package com.bxm.adx.common.market.exchange.rebuild.offer;

import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.buy.dispatcher.Dispatcher;
import com.bxm.adx.common.buy.dispatcher.DispatcherPriceConfig;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.response.Bid;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.annotation.Configuration;

import java.math.BigDecimal;
import java.math.RoundingMode;

/**
 * 媒体出价方式按广告主出价时的实现
 *
 * 1.dsp出价*（1-利润率）保留两位小数
 * 2.计算的出价和配置的cpc上限取最低返回
 *
 * @author jingyu.li
 * @date 2024-04-28
 */
@Slf4j
@Configuration
public class BidByDspPriceOffer implements Offer {

    @Override
    public OfferResult offer(Bid bid, BidRequest request, Dispatcher dispatcher, DispatcherPriceConfig priceConfig) {
        OfferResult.OfferResultBuilder builder = OfferResult.builder().chargeType(AdxConstants.ChargeType.CPC).budgetType(AdxConstants.ChargeType.CPC);
        Integer bidModel = request.getBid_model();
        if (bidModel != AdxConstants.BidModel.SUPPORT_CPC) {
            log.warn("BidByDspPriceOffer bidModel is not CPC！strategyId:{}",dispatcher.getStrategyId());
            return builder.build();
        }

        //dsp出价判断
        BigDecimal bidPrice = bid.getBid();
        if (bidPrice == null) {
            log.warn("BidByDspPriceOffer bidPrice is empty！");
            return builder.build();
        }
        if (bidPrice.compareTo(BigDecimal.ZERO) == 0) {
            log.warn("BidByDspPriceOffer bidPrice = 0！bidId:{}",bid.getId());
            return builder.offer(bidPrice).build();
        }

        //利润率判断
        BigDecimal profitMargin = priceConfig.getProfitMargin();
        if (profitMargin == null) {
            log.warn("BidByDspPriceOffer profitMargin is empty！positionId:{},strategyId:{},priceModeName:{}",dispatcher.getPositionId(),dispatcher.getStrategyId(),priceConfig.getPriceModeName());
            return builder.build();
        }

        //cpc上限判断
        BigDecimal cpcUpperLimit = priceConfig.getCpcUpperLimit();
        if (cpcUpperLimit == null) {
            log.warn("BidByDspPriceOffer cpcUpperLimit is empty！positionId:{},strategyId:{},priceModeName:{}",dispatcher.getPositionId(),dispatcher.getStrategyId(),priceConfig.getPriceModeName());
            return builder.build();
        }

        //按广告主出价方式给媒体出价计算方式： dsp出价*（1-利润率）保留两位小数
        BigDecimal newBidPrice = bidPrice.multiply(new BigDecimal(1).subtract(profitMargin.movePointLeft(2))).setScale(0, RoundingMode.FLOOR);

        return builder.offer(newBidPrice.min(cpcUpperLimit.movePointRight(2)).setScale(2, RoundingMode.FLOOR)).build();
    }

    @Override
    public OfferType offerType() {
        return OfferType.Bid_By_Dsp;
    }
}
