package com.bxm.adx.common.market.exchange.rebuild.response;

import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.adapter.AbstractPluginBuyModelAdapter;
import com.bxm.adx.common.adapter.BuyModelAdapter;
import com.bxm.adx.common.buy.Buyer;
import com.bxm.adx.common.buy.dispatcher.Dispatcher;
import com.bxm.adx.common.buy.dispatcher.DispatcherPriceConfig;
import com.bxm.adx.common.buy.dsp.Dsp;
import com.bxm.adx.common.sell.BidResponse;
import com.bxm.adx.common.sell.response.Bid;
import com.bxm.adx.common.sell.response.ClickMonitor;
import com.bxm.adx.common.sell.response.ImpMonitor;
import com.bxm.adx.common.sell.response.SeatBid;
import com.bxm.adx.facade.constant.enums.AdxErrEnum;
import com.bxm.adx.facade.exception.AdxException;
import com.bxm.warcar.integration.pair.Pair;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * DSP赢价上报处理
 *
 * @author fgf
 * @date 2023/5/15
 **/
@Configuration
@Slf4j
public class DspWinPriceBuilder implements AdxBidResponseBuilder {
    private final String SECOND_AUCTION_DSP_LIST = "adx.second.auction.dsp";
    private final Pair pair;

    public DspWinPriceBuilder(Pair pair) {
        this.pair = pair;
    }

    @Override
    public void rebuildAdxBidResponse(BidResponse bidResponse, ResponseBuildAttribute attribute) {
        Buyer buyer = attribute.getBuyer();
        Dsp dsp = buyer.getDsp();
        String dspKey = dsp.getApiKey();
        BuyModelAdapter adapter = buyer.getModelAdapter();
        if (!(adapter instanceof AbstractPluginBuyModelAdapter)) {
            return;
        }
        AbstractPluginBuyModelAdapter pluginAdapter = (AbstractPluginBuyModelAdapter) adapter;
        String macro = pluginAdapter.getPluginConfig().getProperties().getProperty(AdxConstants.PluginParam.MACRO_WIN_PRICE);
        if (StringUtils.isBlank(macro)) {
            return;
        }
        for (SeatBid seatBid : bidResponse.getSeat_bid()) {
            for (Bid bid : seatBid.getBid()) {
                if (Objects.nonNull(bid.getDspWinPrice())) {
                    continue;
                }
                Integer chargeType = getChargeType(bid);
                BigDecimal dspWinPrice = getWinPrice(chargeType, bid, attribute);
                //校验在指定媒体和DSP下生效
                if (check(attribute)) {
                    if (isSecondAuction(bid, attribute, chargeType, dspWinPrice)) {
                        continue;
                    }
                }
                dspWinPrice = pluginAdapter.priceScale(dspWinPrice);
                bid.setDspWinPrice(dspWinPrice);

                String cipher = pluginAdapter.encrypt(dspWinPrice.toString(), dspKey);
                if (StringUtils.isBlank(cipher)) {
                    continue;
                }
                switch (chargeType) {
                    case AdxConstants.ChargeType.CPM:
                        List<ImpMonitor> impMonitors = bid.getImp_monitors();
                        if (CollectionUtils.isEmpty(impMonitors)) {
                            continue;
                        }
                        impMonitors.forEach(
                                impMonitor -> {
                                    String imp = impMonitor.getImp_monitor_url().replace(macro, cipher);
                                    impMonitor.setImp_monitor_url(imp);
                                }
                        );
                        break;
                    case AdxConstants.ChargeType.CPC:
                        List<ClickMonitor> clickMonitors = bid.getClick_monitors();
                        if (CollectionUtils.isEmpty(clickMonitors)) {
                            continue;
                        }
                        clickMonitors.forEach(
                                clickMonitor -> {
                                    String click = clickMonitor.getClick_monitor_url().replace(macro, cipher);
                                    clickMonitor.setClick_monitor_url(click);
                                }
                        );
                        break;
                }
            }
        }
    }

    @Override
    public int getOrder() {
        return 4;
    }

    private static final List<String> MEDIA_LIST = Lists.newArrayList("1", "42");
    /**
     * 只在媒体bes/bxm和dsp盘古上生效
     * @return
     */
    private boolean check(ResponseBuildAttribute attribute) {
        List<String> dspIds = pair.get(SECOND_AUCTION_DSP_LIST).ofArrayList();
        if (CollectionUtils.isEmpty(dspIds)) {
            return false;
        }
        String id = attribute.getBuyer().getDsp().getId().toString();
        String mediaId = attribute.getBidRequest().getMediaId();
        return dspIds.contains(id) && MEDIA_LIST.contains(mediaId);
    }

    /**
     * 是否二价成交模式
     *
     * @param bid
     * @param attribute
     * @return
     */
    private boolean isSecondAuction(Bid bid, ResponseBuildAttribute attribute, Integer chargeType, BigDecimal dspWinPrice) {
        DispatcherPriceConfig priceConfig = attribute.getBidPriceConfig(bid);
        if (Objects.nonNull(priceConfig)) {
            boolean isSecondAuction = false;
            switch (chargeType) {
                case AdxConstants.ChargeType.CPM:
                    Integer winPriceType = priceConfig.getAdvertiserWinPriceType();
                    isSecondAuction = Objects.nonNull(winPriceType) && DispatcherPriceConfig.ADV_WIN_PRICE_TYPE_MEDIA == winPriceType;
                    if (isSecondAuction) {
                        bid.setAdv_wpt(winPriceType);
                        BigDecimal advertiserProfitMargin = priceConfig.getAdvertiserProfitMargin();
                        if (Objects.isNull(advertiserProfitMargin)) {
                            log.warn("position {} config {} adv-pm is null", priceConfig.getPositionId(), priceConfig.getPositionId());
                        }
                        bid.setAdv_pm(advertiserProfitMargin);
                        bid.setDsp_dis_price(dspWinPrice);
                    }
                    break;
                case AdxConstants.ChargeType.CPC:
                    Integer offerType = priceConfig.getAdvertiserOfferType();
                    isSecondAuction = Objects.nonNull(offerType) && AdxConstants.DispatcherDspWinType.MEDIA_BASE == offerType;
                    if (isSecondAuction) {
                        bid.setAdv_wpt(DispatcherPriceConfig.ADV_WIN_PRICE_TYPE_MEDIA);
                        BigDecimal advertiserProfitMargin = priceConfig.getAdvertiserProfitMargin();
                        if (Objects.isNull(advertiserProfitMargin)) {
                            log.warn("position {} config {} adv-pm is null", priceConfig.getPositionId(), priceConfig.getPositionId());
                        }
                        bid.setAdv_pm(advertiserProfitMargin);
                    }
                    break;
            }
            return isSecondAuction;
        }
        return false;
    }

    /**
     * 获取赢价
     *
     * @param chargeType
     * @param bid
     * @param attribute
     * @return
     */
    private static BigDecimal getWinPrice(Integer chargeType, Bid bid, ResponseBuildAttribute attribute) {
        BigDecimal price = bid.getDsp_price();
        Dispatcher dispatcher = attribute.getDispatcher();
        switch (chargeType) {
            case AdxConstants.ChargeType.CPM:
                return getCpmWinPrice(price, bid, attribute);
            case AdxConstants.ChargeType.CPC:
                return getCpcWinPrice(price, bid, attribute);
        }
        throw new AdxException(AdxErrEnum.UNKNOWN_ERR);
    }

    /**
     * 获取计费类型
     *
     * @param bid
     * @return
     */
    private static Integer getChargeType(Bid bid) {
        Integer chargeType = bid.getCharge_type();
        return Objects.isNull(chargeType) ? AdxConstants.ChargeType.CPM : chargeType;
    }

    /**
     * CPM赢价计算
     * 例子：广告主出价10元，8元以内部分折扣系数0.8，8元以外部分0.4，则上报广告主赢价=8*0.8+（10-8）*0.4
     *
     * @param dspOffer
     * @return
     */
    private static BigDecimal getCpmWinPrice(BigDecimal dspOffer, Bid bid, ResponseBuildAttribute attribute) {
        DispatcherPriceConfig config = attribute.getBidPriceConfig(bid);
        //未配置出价配置直接返回原出价
        if (Objects.isNull(config)) {
            return dspOffer;
        }
        BigDecimal priceSplit = config.getPriceSplit();
        BigDecimal lt = config.getLtPriceDiscountFactor();
        BigDecimal gt = config.getGtPriceDiscountFactor();
        //未配置的时段使用默认系数
        if (Objects.isNull(priceSplit)) {
            lt = Optional.ofNullable(lt).orElse(BigDecimal.ONE);
            return dspOffer.multiply(lt);
        } else {
            priceSplit = priceSplit.movePointRight(2);
            //看注释例子
            int compare = dspOffer.compareTo(priceSplit);
            if (compare <= 0) {
                return dspOffer.multiply(lt);
            } else {
                BigDecimal ltPrice = priceSplit.multiply(lt);
                BigDecimal gtPrice = dspOffer.subtract(priceSplit).multiply(gt);
                return ltPrice.add(gtPrice);
            }
        }
    }

    /**
     * CPC计费时使用固价作为赢价，固价不高于dsp出价
     *
     * @param dspOffer
     * @return
     */
    private static BigDecimal getCpcWinPrice(BigDecimal dspOffer, Bid bid, ResponseBuildAttribute attribute) {
        DispatcherPriceConfig config = attribute.getBidPriceConfig(bid);
        //未配置出价配置直接返回原出价
        if (Objects.isNull(config)) {
            return dspOffer;

        }
        Integer type = config.getAdvertiserOfferType();
        if (Objects.isNull(type)) {
            log.warn("dispatcher {} advert win type is null", attribute.getDispatcher().getId());
            throw new AdxException(AdxErrEnum.DISPATCHER_ERR);
        }
        if (AdxConstants.DispatcherDspWinType.MEDIA_BASE == type) {
            return null;
        }
        BigDecimal price = config.getAdvertiserOfferPrice();
        if (Objects.isNull(price)) {
            log.warn("dispatcher {} advert offer price is empty", attribute.getDispatcher().getId());
            throw new AdxException(AdxErrEnum.DISPATCHER_ERR);
        }
        //单位元转分
        price = price.movePointRight(2);
        if (price.compareTo(dspOffer) > 0) {
            return dspOffer;
        }
        return price;
    }
}
