package com.bxm.adx.common.market.exchange.rebuild.response;

import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.ReplaceCreativeProperties;
import com.bxm.adx.common.buy.dispatcher.Dispatcher;
import com.bxm.adx.common.creative.replace.ReplaceCreative;
import com.bxm.adx.common.creative.replace.ReplaceCreativeDao;
import com.bxm.adx.common.ip.IpService;
import com.bxm.adx.common.market.exchange.ExchangeContext;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.BidResponse;
import com.bxm.adx.common.sell.request.Device;
import com.bxm.adx.common.sell.response.*;
import com.bxm.warcar.ip.IP;
import com.bxm.warcar.utils.JsonHelper;
import com.google.common.collect.Sets;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.RandomUtils;
import org.springframework.context.annotation.Configuration;

import java.net.URI;
import java.time.LocalTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author fgf
 * @date 2023/1/9
 **/
@Configuration
public class NativeBuilder implements AdxBidResponseBuilder {
    private final ReplaceCreativeProperties replaceCreativeProperties;
    private final ReplaceCreativeDao replaceCreativeDao;
    private final IpService ipService;

    public NativeBuilder(ReplaceCreativeProperties replaceCreativeProperties, ReplaceCreativeDao replaceCreativeDao, IpService ipService) {
        this.replaceCreativeProperties = replaceCreativeProperties;
        this.replaceCreativeDao = replaceCreativeDao;
        this.ipService = ipService;
    }

    @Override
    public int getOrder() {
        return 3;
    }

    @Override
    public void rebuildAdxBidResponse(BidResponse bidResponse, ResponseBuildAttribute attribute) {
        BidRequest request = attribute.getBidRequest();
        if (mediaLimit(request)) {
            return;
        }

        Dispatcher dispatcher = attribute.getDispatcher();
        ReplaceCreative creative = getCreative(dispatcher);
        if (Objects.isNull(creative)) {
            return;
        }
        boolean isFullUse = isFullUse(request);
        //处理素材url
        for (SeatBid seatBid : bidResponse.getSeat_bid()) {
            for (Bid bid : seatBid.getBid()) {
                com.bxm.adx.common.sell.response.Native an = bid.getA_native();
                //暂不支持如果预算完全没有素材返回的情况下替换素材
                if (Objects.nonNull(creative) && Objects.nonNull(an) && CollectionUtils.isNotEmpty(an.getAssets())) {
                    List<Asset> replaceAssetList = needReplace(request, creative);
                    if (isFullUse) {
                        an.setAssets(replaceAssetList);
                    } else {
                        //替换素材和DSP素材取并集，素材类型相同时使用替换素材
                        buildNative(an, replaceAssetList);
                    }
                    bid.setAdxCreateId(creative.getId().toString());
                }
            }
        }
    }

    /**
     * 使用完全使用替换素材
     *
     * @return
     */
    private boolean isFullUse(BidRequest request) {
        List<String> mediaIdList = replaceCreativeProperties.getFullUseMediaIdList();
        return CollectionUtils.isNotEmpty(mediaIdList) && mediaIdList.contains(request.getMediaId());
    }

    /**
     * 获取替换素材
     *
     * @param dispatcher
     * @return
     */
    private ReplaceCreative getCreative(Dispatcher dispatcher) {
        List<ReplaceCreative> creatives = queryReplaceCreativeByDispatcher(dispatcher);
        ReplaceCreative creative = randomCreative(creatives);
        return creative;
    }

    /**
     * 随机出替换素材
     *
     * @param creatives
     * @return
     */
    private ReplaceCreative randomCreative(List<ReplaceCreative> creatives) {
        if (CollectionUtils.isNotEmpty(creatives)) {
            int r = RandomUtils.nextInt(creatives.size());
            return creatives.get(r);
        }
        return null;
    }

    /**
     * 开启媒体投放限制
     *
     * @param request
     * @return
     */
    private boolean mediaLimit(BidRequest request) {
        ReplaceCreativeProperties.LimitConfig config = replaceCreativeProperties.getLimitConfig();
        if (Objects.isNull(config)) {
            return false;
        }
        String mediaId = request.getMediaId();
        List<String> mediaIdList = config.getMediaIdList();
        if (CollectionUtils.isNotEmpty(mediaIdList) && mediaIdList.contains(mediaId)) {
            if (timeLimit() && regionalLimit(request, config)) {
                return true;
            }
        }
        return false;
    }

    /**
     * 时间限制
     *
     * @return
     */
    private boolean timeLimit() {
        LocalTime dateTime = LocalTime.now();
        LocalTime startTime = LocalTime.of(9, 30);
        LocalTime endTime = LocalTime.of(18, 30);
        return dateTime.isAfter(startTime) && dateTime.isBefore(endTime);
    }

    /**
     * 地区限制
     *
     * @return
     */
    private boolean regionalLimit(BidRequest request, ReplaceCreativeProperties.LimitConfig config) {
        List<String> provinceCodes = config.getProvinceCodeList();
        if (CollectionUtils.isEmpty(provinceCodes)) {
            return false;
        }
        IP ip = getIp(request);
        if (Objects.isNull(ip)) {
            return false;
        }
        if (ip.isHit()) {
            String hitcode = ip.getHitcode();
            String province = getCodeForProvince(hitcode);
            return provinceCodes.contains(province);
        }
        return false;
    }

    private IP getIp(BidRequest bidRequest) {

        IP ipp = ExchangeContext.getIp();
        if (Objects.nonNull(ipp)) {
            return ipp;
        }
        if (bidRequest != null) {
            Device device = bidRequest.getDevice();
            if (device != null) {
                String ip = device.getIp();
                if (!org.springframework.util.StringUtils.isEmpty(ip)) {
                    ipp = ipService.analyze(ip);
                    return ipp;
                }
            }
        }
        return ipp;
    }

    private String getCodeForProvince(String code) {
        return StringUtils.rightPad(StringUtils.left(code, 2), 6, "0");
    }

    /**
     * 获取所有替换素材
     *
     * @param dispatcher
     * @return
     */
    private List<ReplaceCreative> queryReplaceCreativeByDispatcher(Dispatcher dispatcher) {
        List<Long> creativeIds = dispatcher.getCreativeIdList();
        if (CollectionUtils.isNotEmpty(creativeIds)) {
            List<ReplaceCreative> creativeList = replaceCreativeDao.queryCreativeListByIds(creativeIds);
            return creativeList;
        }
        return null;
    }

    private String urlRequestTimeAuth(BidRequest request, String url) {
        ReplaceCreativeProperties.ErConfig erConfig = replaceCreativeProperties.getErConfig();
        if (Objects.isNull(erConfig)) {
            return url;
        }

        String mediaId = request.getMediaId();
        List<String> mediaIdList = erConfig.getMediaIdList();
        if (CollectionUtils.isEmpty(mediaIdList) || !(mediaIdList.contains(mediaId))) {
            return url;
        }

        String host = erConfig.getNewHost();
        if (StringUtils.isBlank(host)) {
            return url;
        }
        return getNewUrl(url, host);
    }

    /**
     * 替换素材url，配合阿里云DCDN的边缘程序功能实现url访问时效（24小时内）
     * 原url: https://buyimg.bianxianmao.com/2022/04/20/b68cd42080444305945ee8a0d1f09b48.jpeg
     * 替换后: https://adximg.bianxianmao.com/2022/04/20/b68cd42080444305945ee8a0d1f09b48-1686549745.jpeg
     *
     * @param url
     * @return
     */
    private String getNewUrl(String url, String newHost) {
        if (StringUtils.isNotBlank(url)) {
            URI uri = URI.create(url);
            String path = uri.getPath();
            long timestamp = System.currentTimeMillis() / 1000;
            path = path.replace(".", "-" + timestamp + ".");
            return newHost + path;
        }
        return url;
    }

    /**
     * 替换DSP的素材
     *
     * @param an
     * @param needReplace
     * @return
     */
    private void buildNative(com.bxm.adx.common.sell.response.Native an, List<Asset> needReplace) {
        if (CollectionUtils.isEmpty(needReplace)) {
            return;
        }
        Set<Integer> replacedTypes = Sets.newHashSet();
        List<com.bxm.adx.common.sell.response.Asset> newAsset = new ArrayList<>();
        for (com.bxm.adx.common.sell.response.Asset asset : an.getAssets()) {
            Integer type = asset.getType();
            if (Objects.isNull(type)) {
                continue;
            }
            Optional<Asset> replace = null;
            if (type.equals(AdxConstants.AssetType.REWARDER_VIDEO.getType())) {
                replace = needReplace.stream()
                        .filter(a -> a.getType().equals(AdxConstants.AssetType.VIDEO.getType()))
                        .findFirst();
            } else {
                replace = needReplace.stream()
                        .filter(a -> a.getType().equals(type))
                        .findFirst();
            }
            if (replace.isPresent()) {
                com.bxm.adx.common.sell.response.Asset replaceAsset = replace.get();
                replacedTypes.add(replaceAsset.getType());
                if (type.equals(AdxConstants.AssetType.VIDEO.getType()) || type.equals(AdxConstants.AssetType.REWARDER_VIDEO.getType())) {
                    Video oldVideo = asset.getVideo();
                    Video newVideo = replaceAsset.getVideo();
                    if (Objects.nonNull(oldVideo)) {
                        newVideo.setV_monitor(oldVideo.getV_monitor());
                        if (Objects.nonNull(newVideo.getCover_url())) {
                            oldVideo.setCover_url(newVideo.getCover_url());
                        }
                        if (Objects.nonNull(newVideo.getW())) {
                            oldVideo.setW(newVideo.getW());
                        }
                        if (Objects.nonNull(newVideo.getH())) {
                            oldVideo.setH(newVideo.getH());
                        }
                        if (Objects.nonNull(newVideo.getDuration())) {
                            oldVideo.setDuration(newVideo.getDuration());
                        }
                        if (Objects.nonNull(newVideo.getMax_length())) {
                            oldVideo.setMax_length(newVideo.getMax_length());
                        }
                        if (Objects.nonNull(newVideo.getUrl())) {
                            oldVideo.setUrl(newVideo.getUrl());
                        }
                        if (Objects.nonNull(newVideo.getMime_types())) {
                            oldVideo.setMime_types(newVideo.getMime_types());
                        }
                    }
                    com.bxm.adx.common.sell.response.Asset newVideoAsset = com.bxm.adx.common.sell.response.Asset.builder()
                            .type(type)
                            .video(oldVideo)
                            .build();
                    newAsset.add(newVideoAsset);
                    continue;
                }
                // FIXME: 2022/11/24 FastJson对重复/循环引用的处理
                com.bxm.adx.common.sell.response.Asset copyAsset = JsonHelper.convert(JsonHelper.convert(replaceAsset), com.bxm.adx.common.sell.response.Asset.class);
                newAsset.add(copyAsset);
            } else {
                newAsset.add(asset);
            }
        }
        List<com.bxm.adx.common.sell.response.Asset> needAdd = needReplace.stream()
                .filter(asset -> !replacedTypes.contains(asset.getType()))
                .collect(Collectors.toList());
        if (CollectionUtils.isNotEmpty(needAdd)) {
            newAsset.addAll(needAdd);
        }
        an.setAssets(newAsset);
    }

    /**
     * 根据替换素材决定具体需要替换的素材元素
     *
     * @param creative
     * @return
     */
    private List<com.bxm.adx.common.sell.response.Asset> needReplace(BidRequest request, ReplaceCreative creative) {
        List<com.bxm.adx.common.sell.response.Asset> assets = new ArrayList<>();
        Integer type = creative.getType();

        String title = creative.getTitle();
        String content = creative.getContent();
        String img = urlRequestTimeAuth(request, creative.getImageUrl());
        String icon = creative.getIconUrl();
        String source = creative.getSource();

        if (StringUtils.isNotEmpty(title)) {
            com.bxm.adx.common.sell.response.Asset titleAsset =
                    com.bxm.adx.common.sell.response.Asset.builder()
                            .type(AdxConstants.AssetType.TITLE.getType())
                            .text(com.bxm.adx.common.sell.response.Text.builder()
                                    .text(title)
                                    .build())
                            .build();
            assets.add(titleAsset);
        }
        if (StringUtils.isNotEmpty(content)) {
            com.bxm.adx.common.sell.response.Asset contentAsset =
                    com.bxm.adx.common.sell.response.Asset.builder()
                            .type(AdxConstants.AssetType.CONTENT.getType())
                            .text(com.bxm.adx.common.sell.response.Text.builder()
                                    .text(content)
                                    .build())
                            .build();
            assets.add(contentAsset);
        }
        if (StringUtils.isNotEmpty(source)) {
            com.bxm.adx.common.sell.response.Asset brandNameAsset =
                    com.bxm.adx.common.sell.response.Asset.builder()
                            .type(AdxConstants.AssetType.BRAND_NAME.getType())
                            .text(com.bxm.adx.common.sell.response.Text.builder()
                                    .text(source)
                                    .build())
                            .build();
            assets.add(brandNameAsset);
        }

        if (StringUtils.isNotEmpty(img)) {
            String imageSize = creative.getImageSize();
            String[] sizes = null;
            Integer w = null;
            Integer h = null;
            if (StringUtils.isNotEmpty(imageSize)) {
                try {
                    sizes = imageSize.split("\\*");
                    w = Integer.valueOf(sizes[0]);
                    h = Integer.valueOf(sizes[1]);
                } catch (Exception e) {
                }
            }
            if (type.equals(ReplaceCreative.TYPE_IMG)) {
                com.bxm.adx.common.sell.response.Asset imgAsset =
                        com.bxm.adx.common.sell.response.Asset.builder()
                                .type(AdxConstants.AssetType.LARGE_IMG.getType())
                                .img(com.bxm.adx.common.sell.response.Image.builder()
                                        .h(h)
                                        .w(w)
                                        .url(img)
                                        .build())
                                .build();
                assets.add(imgAsset);
            }
        }
        if (StringUtils.isNotEmpty(icon)) {
            String iconSize = creative.getIconSize();
            String[] sizes = null;
            Integer w = null;
            Integer h = null;
            if (StringUtils.isNotEmpty(iconSize)) {
                try {
                    sizes = iconSize.split("\\*");
                    w = Integer.valueOf(sizes[0]);
                    h = Integer.valueOf(sizes[1]);
                } catch (Exception e) {
                }
            }
            com.bxm.adx.common.sell.response.Asset iconAsset =
                    com.bxm.adx.common.sell.response.Asset.builder()
                            .type(AdxConstants.AssetType.ICON.getType())
                            .img(com.bxm.adx.common.sell.response.Image.builder()
                                    .h(h)
                                    .w(w)
                                    .url(icon)
                                    .build())
                            .build();
            assets.add(iconAsset);
        }

        if (type.equals(ReplaceCreative.TYPE_VIDEO)) {

            String videoSize = creative.getVideoSize();
            String[] sizes = null;
            Integer w = null;
            Integer h = null;
            if (StringUtils.isNotEmpty(videoSize)) {
                try {
                    sizes = videoSize.split("\\*");
                    w = Integer.valueOf(sizes[0]);
                    h = Integer.valueOf(sizes[1]);
                } catch (Exception e) {
                }
            }
            Video video = Video.builder()
                    .w(w)
                    .h(h)
                    .duration(Objects.isNull(creative.getVideoTime()) ? null : creative.getVideoTime() * 1000)
                    .cover_url(img)
                    .mime_types(StringUtils.isNotBlank(creative.getVideoFormat()) ? Arrays.asList(creative.getVideoFormat()) : null)
                    .url(creative.getVideoUrl())
                    .max_length(Objects.nonNull(creative.getVideoKb()) ? creative.getVideoKb().intValue() : null)
                    .build();
            com.bxm.adx.common.sell.response.Asset videoAsset = com.bxm.adx.common.sell.response.Asset.builder()
                    .video(video)
                    .type(AdxConstants.AssetType.VIDEO.getType())
                    .build();
            assets.add(videoAsset);
        }

        return assets;
    }
}
