package com.bxm.adx.common.market.filter;

import com.bxm.adx.common.AdxConstants;
import com.bxm.adx.common.adapter.AdxContextFactory;
import com.bxm.adx.common.buy.Buyer;
import com.bxm.adx.common.market.Deal;
import com.bxm.adx.common.sell.BidRequest;
import com.bxm.adx.common.sell.BidResponse;
import com.bxm.adx.common.sell.response.Bid;
import com.bxm.adx.common.sell.response.ClickMonitor;
import com.bxm.adx.common.sell.response.ImpMonitor;
import com.bxm.adx.common.sell.response.SeatBid;
import com.bxm.warcar.integration.pair.Pair;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;

import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * adx-bes, 域名白名单过滤
 *
 * @author fgf
 * @date 2023/3/1
 **/
@Slf4j
@Configuration
public class BesDomainFilter implements Filter {
    private final Pair pair;
    private final static String KEY = "adx.bes.domain.list";

    public BesDomainFilter(Pair pair) {
        this.pair = pair;
    }

    @Override
    public void filter(List<Deal> deals, Set<Deal> trash) {
        BidRequest request = AdxContextFactory.get().getBidRequest();
        String mediaId = request.getMediaId();
        if (StringUtils.isEmpty(mediaId) || !mediaId.equals(String.valueOf(AdxConstants.Media.Bes.getId()))) {
            return;
        }
        Set<String> domainSet = pair.get(KEY).ofHashSet();
        if (CollectionUtils.isEmpty(domainSet)) {
            return;
        }

        Set<Deal> remove = deals.stream().filter(
                deal -> {
                    BidResponse response = deal.getBidResponse();
                    List<SeatBid> seatBidList = response.getSeat_bid();

                    seatBidList.removeIf(seatBid -> {
                        List<Bid> bidList = seatBid.getBid();
                        bidList.removeIf(bid -> {
                            return checkDomain(getUrlList(bid), domainSet);
                        });
                        return CollectionUtils.isEmpty(bidList);
                    });
                    return CollectionUtils.isEmpty(seatBidList);
                }
        ).collect(Collectors.toSet());

        if (CollectionUtils.isNotEmpty(remove)) {
            trash.addAll(remove);
        }
    }

    @Override
    public int getOrder() {
        return Ordered.LOWEST_PRECEDENCE;
    }

    private List<String> getUrlList(Bid bid) {
        List<String> urls = Lists.newArrayList();

        List<ImpMonitor> impMonitors = bid.getImp_monitors();
        if (CollectionUtils.isNotEmpty(impMonitors)) {
            urls.addAll(impMonitors.stream().map(ImpMonitor::getImp_monitor_url).collect(Collectors.toList()));
        }

        List<ClickMonitor> clickMonitors = bid.getClick_monitors();
        if (CollectionUtils.isNotEmpty(clickMonitors)) {
            urls.addAll(clickMonitors.stream().map(ClickMonitor::getClick_monitor_url).collect(Collectors.toList()));
        }

        String clickThroughUrl = bid.getClick_through_url();
        if (StringUtils.isNotEmpty(clickThroughUrl)) {
            urls.add(clickThroughUrl);
        }

        return urls;
    }

    private boolean checkDomain(List<String> urls, Set<String> domains) {
        for (String url : urls) {
            String host = null;
            try {
                host = URI.create(url).getHost();
            } catch (Exception e) {
                continue;
            }

            if (StringUtils.isNotEmpty(host)) {
                boolean b = true;
                for (String domain : domains) {
                    if (host.contains(domain)) {
                        b = false;
                        break;
                    }
                }
                if (b) {
                    return true;
                }
            }
        }
        return false;
    }
}
