package com.bxm.adx.common.plugin;


import com.bxm.warcar.dpl2.PluginBus;
import com.bxm.warcar.dpl2.PluginRuntimeException;
import com.bxm.warcar.dpl2.dir.PluginChangedEvent;
import com.bxm.warcar.dpl2.dir.PluginChangedListener;
import com.bxm.warcar.dpl2.plugin.Plugin;
import com.bxm.warcar.dpl2.plugin.PluginConfig;
import com.bxm.warcar.dpl2.plugin.PluginLoader;
import com.bxm.warcar.utils.JsonHelper;
import com.bxm.warcar.utils.NamedThreadFactory;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.*;

/**
 * @author fgf
 * @date 2023/12/26
 * @since 1.0.0
 */
@Slf4j
public class AdxPluginChangedListener implements PluginChangedListener {

    private final PluginBus pluginBus;
    private final PluginLoader pluginLoader;

    private final Map<String, String> filePathAndKeyMapping = Maps.newHashMap();

    private final ExecutorService controlExecutor = new ThreadPoolExecutor(1, 1, 0, TimeUnit.MILLISECONDS,
            new LinkedBlockingQueue<>(),
            new NamedThreadFactory("DPL-Controller"));

    public AdxPluginChangedListener(PluginBus pluginBus, PluginLoader pluginLoader) {
        this.pluginBus = pluginBus;
        this.pluginLoader = pluginLoader;
    }

    @Override
    public void onChanged(PluginChangedEvent event) {
        List<String> added = event.getAdded();
        List<String> updated = event.getUpdated();
        List<String> removed = event.getRemoved();
        List<String> all = event.getAll();

        if (CollectionUtils.isNotEmpty(added)) {
            for (String f : added) {
                String pluginKey = filePathAndKeyMapping.get(f);
                if (StringUtils.isBlank(pluginKey)) {
                    log.info("Plugin file[{}] load has failed previously, add plugin directly.", f);
                    addPlugin(f);
                }
            }
        }
        if (CollectionUtils.isNotEmpty(updated)) {
            for (String f : updated) {
                String pluginKey = filePathAndKeyMapping.get(f);
                if (StringUtils.isBlank(pluginKey)) {
                    log.debug("Plugin file[{}] load has failed previously, add plugin directly.", f);
                    addPlugin(f);
                    continue;
                }
                updatePlugin(f, pluginKey);
            }
        }
        if (CollectionUtils.isNotEmpty(removed)) {
            for (String f : removed) {
                String pluginKey = filePathAndKeyMapping.get(f);
                if (StringUtils.isBlank(pluginKey)) {
                    continue;
                }
                removePlugin(f, pluginKey);
            }
        }

        if (filePathAndKeyMapping.size() != all.size()) {
            Set<String> installed = filePathAndKeyMapping.keySet();
            List<String> diff = new ArrayList<>(all);
            diff.removeAll(installed);
            log.error("installPlugIn fail[" + diff + "]");
            throw launderThrowable("installPlugin fail", new RuntimeException("installed size["+ filePathAndKeyMapping.size() + "]"));
        }
    }

    private void addPlugin(String filePath) {
        String pluginKey = installPlugin(filePath);
        if (StringUtils.isBlank(pluginKey)) {
            return;
        }
        filePathAndKeyMapping.put(filePath, pluginKey);
    }

    private String installPlugin(String filePath) {
        Future<String> future = controlExecutor.submit(new Callable<String>() {
            @Override
            public String call() throws Exception {
                try {
                    Plugin plugin = createPlugin(filePath);
                    if (null == plugin) {
                        return null;
                    }
                    pluginBus.register(plugin);
                    return plugin.getConfig().getKey();
                } catch (PluginRuntimeException e) {
                    log.error("installPlugin: " + filePath, e);
                    return null;
                }
            }
        });
        try {
            return future.get();
        } catch (InterruptedException e) {
            future.cancel(true);
            throw new RuntimeException("installPlugin: ", e);
        } catch (ExecutionException e) {
            future.cancel(true);
            Throwable cause = e.getCause();
            throw launderThrowable("installPlugIn[" + filePath + "]:", cause);
        }
    }

    private Plugin createPlugin(String filePath) {
        return pluginLoader.load(new PluginConfig(filePath));
    }

    private void updatePlugin(String filePath, String pluginKey) {
        uninstallPlugin(pluginKey);
        installPlugin(filePath);
    }

    private void removePlugin(String filePath, String pluginKey) {
        uninstallPlugin(pluginKey);
        filePathAndKeyMapping.remove(filePath);
    }

    private void uninstallPlugin(String pluginKey) {
        controlExecutor.execute(() -> {
            Plugin plugin = pluginBus.lookup(pluginKey);
            if (null == plugin) {
                log.error("Key[{}] is not mapping any plugin.", pluginKey);
                return;
            }

            pluginBus.unregister(plugin);
            plugin.destroy();
        });
    }

    private RuntimeException launderThrowable(String message, Throwable t) {
        if (t instanceof RuntimeException) {
            // return (RuntimeException) t;
            return new RuntimeException(message, t);
        } else if (t instanceof Error) {
            throw (Error) t;
        } else {
            throw new RuntimeException(message, t);
        }
    }
}
