package com.bxm.adx.common.report;

import com.bxm.adx.common.AdxProperties;
import com.bxm.adx.common.CacheKeys;
import com.bxm.adx.common.ServerReportProperties;
import com.bxm.adx.common.openlog.event.internal.DspWinEvent;
import com.bxm.openlog.sdk.KeyValueMap;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.integration.eventbus.EventPark;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.http.message.BasicHeader;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.util.StringUtils;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.Pipeline;

import java.math.BigDecimal;
import java.util.List;
import java.util.Objects;
import java.util.function.Consumer;

import com.bxm.adx.common.AdxConstants.ChargeType;
import com.bxm.adx.common.AdxConstants.BidModel;

/**
 * @author fgf
 * @date 2023/4/19
 **/
@Slf4j
public class ServerReportService {
    private final JedisPool jedisPool;
    private final ReportClient reportClient;
    private final ServerReportProperties serverReportProperties;
    private final AdxProperties properties;
    private final EventPark eventPark;
    private final DspWinPriceHandler dspWinPriceHandler;

    public ServerReportService(ServerReportProperties serverReportProperties, JedisPool jedisPool,
                               ReportClient reportClient, AdxProperties properties, EventPark eventPark, DspWinPriceHandler dspWinPriceHandler) {
        this.serverReportProperties = serverReportProperties;
        this.jedisPool = jedisPool;
        this.reportClient = reportClient;
        this.properties = properties;
        this.eventPark = eventPark;
        this.dspWinPriceHandler = dspWinPriceHandler;
    }

    /**
     * 上报监测链接
     *
     * @param reportParam
     * @param headers
     */
    public void report(ReportParam reportParam, MultiValueMap<String, String> headers, KeyValueMap map) {
        Integer chargeType = reportParam.getChargeType();
        ReportType type = reportParam.getType();
        String bidId = reportParam.getBidId();
        String adId = reportParam.getAdId();
        String dspPriceKey = properties.getCipherProperties().getDspPriceKey();
        BigDecimal win = reportParam.getDspWinPrice(dspPriceKey);
        String dspId = reportParam.getDspId();

        KeyGenerator keyGenerator = null;
        if (Objects.isNull(chargeType)) {
            keyGenerator = getKeyGenerator(type, bidId);
        } else {
            switch (chargeType) {
                case ChargeType.CPC:
                    if (StringUtils.isEmpty(adId)) {
                        keyGenerator = getKeyGenerator(type, bidId);
                    } else {
                        keyGenerator = getKeyGenerator(type, bidId, adId);
                    }
                    break;
                default:
                    keyGenerator = getKeyGenerator(type, bidId);
            }
        }

        DspWinInfo dspWinInfo = null;
        if (Objects.nonNull(win)) {
            dspWinInfo = dspWinPriceHandler.handle(win, Long.valueOf(dspId));
            if (Objects.isNull(dspId)) {
                log.warn("dsp win info is null, bidid = {} type = {}", bidId, type.name());
            }
        }
        int success = report(keyGenerator.generateKey(), headers, dspWinInfo);
        if (success == 0) {
            log.warn("report param {} empty", reportParam);
        }
        //dsp赢价埋点
        eventPark.post(new DspWinEvent(this, win, new LinkedMultiValueMap<>(map), success > 0));
    }

    /**
     * 上报缓存的链接
     *
     * @param key
     * @param headers
     */
    private int report(String key, MultiValueMap<String, String> headers, DspWinInfo info) {
        BasicHeader[] basicHeaders = convertToBasicHeaders(headers);
        int success = 0;
        for (; ; ) {
            try (Jedis jedis = jedisPool.getResource()) {
                jedis.select(serverReportProperties.getQueueDb());
                String url = jedis.lpop(key);
                if (StringUtils.isEmpty(url)) {
                    break;
                }
                if (Objects.nonNull(info)) {
                    url = url.replace(info.getMacro(), info.getCipher());
                }
                if (log.isDebugEnabled()) {
                    log.debug("report url {}", url);
                }
                reportClient.asyncRequest(url, new Consumer<ReportFallback>() {
                    @Override
                    public void accept(ReportFallback reportFallback) {
                        Exception exception = reportFallback.getException();
                        log.warn("report fail {} url {}, headers {}",
                                exception == null ? "" : exception.getMessage(), reportFallback.getUrl(), reportFallback.getHeaders());
                    }
                }, basicHeaders);
                success++;
            } catch (Exception e) {
                log.error("report err {}, affect {} {}", e.getMessage(), key, headers);
            }
        }
        return success;
    }

    /**
     * 缓存监测链接
     *
     * @param reportParam
     * @param urls
     */
    public void saveExceed(ReportParam reportParam, List<String> urls) {
        if (CollectionUtils.isEmpty(urls)) {
            return;
        }

        KeyGenerator keyGenerator = null;
        Integer bidModel = reportParam.getBidModel();
        ReportType type = reportParam.getType();
        String bidId = reportParam.getBidId();
        String adId = reportParam.getAdId();
        switch (bidModel) {
            case BidModel.SUPPORT_CPC:
                keyGenerator = getKeyGenerator(type, bidId, adId);
                break;
            default:
                keyGenerator = getKeyGenerator(type, bidId);
        }

        save(keyGenerator.generateKey(), urls);
    }

    /**
     * 缓存服务端上报的监测
     *
     * @param urls
     */
    private void save(String key, List<String> urls) {
        try (Jedis jedis = jedisPool.getResource()) {
            try (Pipeline pipeline = jedis.pipelined()) {
                pipeline.select(serverReportProperties.getQueueDb());
                pipeline.lpush(key, urls.toArray(new String[0]));
                pipeline.expire(key, serverReportProperties.getExpire());
                List<Object> response = pipeline.syncAndReturnAll();
                if (log.isDebugEnabled()) {
                    log.debug("pipeline response {}", response);
                }
            }
        } catch (Exception e) {
            log.error("save exceed err {}, affect {}", e.getMessage(), urls);
        }
    }

    /**
     * 获取存储的key
     *
     * @param type
     * @param id
     * @return
     */
    public KeyGenerator getKeyGenerator(ReportType type, String id) {
        return CacheKeys.ServerReport.getServerReportTempKey(type.name(), id);
    }

    public KeyGenerator getKeyGenerator(ReportType type, String id, String adId) {
        return CacheKeys.ServerReport.getServerReportTempKey(type.name(), id, adId);
    }

    public ServerReportConfig getConfig(String mediaId, String appId, String dspId, String bidId) {
        ServerReportProperties.MediaConfig mediaConfig = getMedia(mediaId, appId);
        if (Objects.isNull(mediaConfig)) {
            return null;
        }
        if (!serverReportProperties.getDspIds().contains(dspId)) {
            return null;
        }

        return ServerReportConfig.builder()
                .limit(mediaConfig.getNum() - 1)
                .bidId(bidId)
                .domains(serverReportProperties.getDomains())
                .appId(appId)
                .dspId(dspId)
                .mediaId(mediaId)
                .build();
    }

    /**
     * 匹配媒体
     *
     * @param mediaId
     * @param appId
     * @return
     */
    private ServerReportProperties.MediaConfig getMedia(String mediaId, String appId) {
        List<ServerReportProperties.MediaConfig> media = serverReportProperties.getMedia();
        for (ServerReportProperties.MediaConfig config : media) {
            if (mediaId.equals(config.getId())) {
                List<String> appIds = config.getAppIds();
                if (CollectionUtils.isNotEmpty(appIds)) {
                    if (appIds.contains(appId)) {
                        return config;
                    }
                } else {
                    return config;
                }
            }
        }
        return null;
    }

    public BasicHeader[] convertToBasicHeaders(MultiValueMap<String, String> map) {
        BasicHeader[] headers = new BasicHeader[map.size()];
        int i = 0;
        for (String key : map.keySet()) {
            String[] values = map.get(key).toArray(new String[0]);
            headers[i] = new BasicHeader(key, String.join(", ", values));
            i++;
        }
        return headers;
    }

    @Data
    @Builder
    @AllArgsConstructor
    @NoArgsConstructor
    static class Request {
        private String url;
        private BasicHeader[] headers;
    }
}
