package com.bxm.adx.timer.service;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.adx.timer.common.MediaEntranceCreativesBuider;
import com.bxm.adx.timer.common.OpenClickModel;
import com.bxm.warcar.algorithm.NumericalModel;
import com.bxm.warcar.algorithm.RequestModel;
import com.bxm.warcar.algorithm.flow.service.FlowAlgorithmService;
import com.bxm.warcar.algorithm.utils.DoubleUtils;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.utils.NamedThreadFactory;
import com.bxm.warcar.utils.localdate.LocalDateTimeHelper;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * <h3>类的基本描述</h3>
 *
 * @author hcmony
 * @since V1.0.0, 2020/8/6 14:10
 */
@Service
public class MediaEntranceCreativesServiceImpl implements MediaEntranceCreativesService {

	private final static Logger LOGGER = LoggerFactory.getLogger(MediaEntranceCreativesService.class);

	private LoadingCache<String, Map<String, String>> positionPvCache =
			CacheBuilder.newBuilder()
					.refreshAfterWrite(1, TimeUnit.SECONDS)//每小时刷新缓存
					.build(new CacheLoader<String, Map<String, String>>() {
						@Override
						public Map<String, String> load(String key) {
							return fetcher.hfetchall(
									MediaEntranceCreativesBuider.positionPrePv(LocalDateTimeHelper.preDateString()),String.class);
						}
					});

	private final static ExecutorService pool = new ThreadPoolExecutor(
			Runtime.getRuntime().availableProcessors(),
			Runtime.getRuntime().availableProcessors(),
			0L,
			TimeUnit.MINUTES,
			new LinkedBlockingQueue<>(),
			new NamedThreadFactory("positionMaterial")
	);

	@Autowired
	private FlowAlgorithmService flowAlgorithmService;

	@Qualifier("jedisFetcher")
	@Autowired
	private Fetcher fetcher;

	@Qualifier("jedisUpdater")
	@Autowired
	private Updater updater;

	/**
	 * 素材流量计算分配
	 */
	@Override
	public void flowAlgorithm() throws Exception {

		//1,查询最近小时有数据的广告位
		long startTime = System.currentTimeMillis();

		//2,查询广告位昨日曝光PV
		final Map<String, String> positionPvMap;
		if (positionPvCache.size()>0){
			positionPvMap = (Map<String, String>) positionPvCache.get("");
		}else {
			positionPvMap = fetcher.hfetchall(
					MediaEntranceCreativesBuider.positionPrePv(LocalDateTimeHelper.preDateString()), String.class);
		}
		if (MapUtils.isEmpty(positionPvMap)) {
			LOGGER.warn("查询所有广告位昨日曝光PV数据为空");
			return;
		}

		//3,广告位最近24小时点击率（点击/曝光）
		Map<String, String> rateMap = fetcher.hfetchall(MediaEntranceCreativesBuider.positionClickRate24H(), String.class);
		if (MapUtils.isEmpty(rateMap)) {
			LOGGER.warn("查询所有广告位最近24小时点击率数据为空");
			return;
		}

		Set<Map.Entry<String, String>> entries = rateMap.entrySet();
		List<Future<Integer>> futures = new ArrayList<>(rateMap.size());
		AtomicInteger atomicInteger = new AtomicInteger(1);
		for (Map.Entry<String, String> entry : entries) {
			Future<Integer> future = pool.submit(() -> {
				doFlow(positionPvMap, entry);
				return atomicInteger.getAndIncrement();
			});
			futures.add(future);
		}

		futures.forEach(future -> {
			try {
				int index = future.get();
				if (index==1 || index % 100 == 0 || index == futures.size()) {
					LOGGER.info("Update material flow  {} / {} .", index, futures.size());
				}
			} catch (Exception e) {
				LOGGER.error(e.getMessage(),e);
			}
		});

		LOGGER.info("Update material flow finished {} s ", DoubleUtils.divide(System.currentTimeMillis() - startTime, 1000));
	}

	private void doFlow(Map<String, String> positionPvMap, Map.Entry<String, String> entry) {

		final String positionId = entry.getKey();
		final String positionValue = entry.getValue();
		if (StringUtils.isBlank(positionValue)) {
			return;
		}
		//广告位最近24小时曝光,点击,点击率
		final OpenClickModel positionModel = JSON.parseObject(positionValue, OpenClickModel.class);

		//4,去redis获取广告位上所有的素材
		final List<String> materialIds = fetchMaterialIds(positionId);
		if (CollectionUtils.isEmpty(materialIds)) {
			LOGGER.warn("position : {} material is empty . ",positionId);
			return;
		}
		List<NumericalModel> numericalModels = new ArrayList<>(materialIds.size());

		//5,广告位前日pv处理
		final String preGroupDatePvString = positionPvMap.get(positionId);
		long  preGroupDatePv = 0 ;
		if (StringUtils.isNotBlank(preGroupDatePvString)){
			preGroupDatePv = Long.parseLong(preGroupDatePvString);
		}

		//6,循环处理每个素材模型曝光,点击,点击率,2h,24h,等 数据
		for (String id : materialIds) {
			numericalModels.add(convertNumericalModel(id, positionId, positionModel, preGroupDatePv));
		}

		//7,数据计算分配
		RequestModel<NumericalModel> requestModel = new RequestModel<NumericalModel>(numericalModels);
		requestModel = flowAlgorithmService.deServie(requestModel);

		//写入缓存
		writeRedis(requestModel, positionId);
	}

	/**
	 * NumericalModel(String id, String groupId, Long openPv, Long preGroupDatePv, Double groupClickRate,
	 *  Double clickRate2H,Double clickRate24H ,Double clickRate)
	 * @param id
	 * @param positionId
	 * @param positionModel
	 * @return
	 */
	private NumericalModel convertNumericalModel(String id,String positionId,OpenClickModel positionModel,long preGroupDatePv) {

		NumericalModel model = new NumericalModel(id,positionId,0L,preGroupDatePv,
											positionModel.getClickRate(),0.0);
		try {
			//1，两小时
			String clickRate2h = fetcher.hfetch(MediaEntranceCreativesBuider.clickRate2H(positionId),id,String.class);
			if (StringUtils.isNotBlank(clickRate2h)){
				OpenClickModel clickRate2hModel = JSON.parseObject(clickRate2h,OpenClickModel.class);
				model.setClickRate2H(clickRate2hModel.getClickRate());
				model.setOpenPv2H(clickRate2hModel.getExposurePv());
			}

			//2，24小时
			String clickRate24h = fetcher.hfetch(MediaEntranceCreativesBuider.clickRate24H(positionId),id,String.class);
			if (StringUtils.isNotBlank(clickRate24h)){
				OpenClickModel clickRate24hModel = JSON.parseObject(clickRate24h,OpenClickModel.class);
				model.setClickRate24H(clickRate24hModel.getClickRate());
				model.setOpenPv24H(clickRate24hModel.getExposurePv());
			}

			//3，历史
			String materialclickRate = fetcher.hfetch(MediaEntranceCreativesBuider.clickRate(positionId),id,String.class);
			if (StringUtils.isNotBlank(materialclickRate)){
				OpenClickModel materialclickRateModel = JSON.parseObject(materialclickRate,OpenClickModel.class);
				model.setClickRate(materialclickRateModel.getClickRate());
				model.setOpenPv(materialclickRateModel.getExposurePv());
			}
			return model;
		}catch (Exception e){
			LOGGER.error(e.getMessage(),e);
			return model;
		}
	}


	private List<String> fetchMaterialIds(String positionId) {
		String ids = fetcher.hfetch(MediaEntranceCreativesBuider.positionAllCreatives(),positionId, String.class);
		if (StringUtils.isNotBlank(ids)) {
			return JSONObject.parseArray(ids, String.class);
		}
		return null;
	}

	private void writeRedis(RequestModel<NumericalModel> requestModel, String positionId) {
		if (requestModel == null) {
			return;
		}
		Map<String, Double> map = requestModel.getMap();
		if (MapUtils.isEmpty(map)) {
			return;
		}
		updater.update(MediaEntranceCreativesBuider.flowRate(positionId), JSONObject.toJSONString(map));
	}

}
