package com.bxm.datapark.service.count;

import com.bxm.datapark.constant.Constant;
import com.bxm.datapark.dal.model.CountAwradmsgDetail;
import com.bxm.datapark.dal.model.CountTicketAdvertiser;
import com.bxm.datapark.dal.model.ext.MqResult;
import com.bxm.datapark.service.service.CountService;
import com.bxm.warcar.mq.ConsumeStatus;
import com.bxm.warcar.mq.Message;
import com.bxm.warcar.mq.SingleMessageListener;
import com.bxm.warcar.utils.JsonHelper;
import org.apache.commons.lang.ArrayUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <h3>Abstract count message listener</h3>
 *
 * @author allen
 * @since 1.0.0
 */
public abstract class AbstractCountMessageListener implements SingleMessageListener {

    private static final Logger LOGGER = LoggerFactory.getLogger(AbstractCountMessageListener.class);

    protected abstract CountService getCountService();

    @Override
    public ConsumeStatus consume(Message message, Object context) {
        CountService service = getCountService();
        if (null == service) {
            if (LOGGER.isErrorEnabled()) {
                LOGGER.error("service is null, ignored this message.", message.getMsgId());
            }
            return ConsumeStatus.CONSUME_SUCCESS;
        }

        if (message.getReconsumeTimes() >= Constant.MAX_RETRY_TIMES) {
            if (LOGGER.isWarnEnabled()) {
                LOGGER.warn("[{}] message [{}] has retried more than 3 times, it will be ignored.", getClass().getName(), message.getMsgId());
            }
            return ConsumeStatus.CONSUME_SUCCESS;
        }

        byte[] body = message.getBody();
        if (ArrayUtils.isEmpty(body)) {
            return ConsumeStatus.CONSUME_SUCCESS;
        }

        MqResult result = null;
        try {
            result = JsonHelper.convert(body, MqResult.class);
        } catch (Exception e) {
            if (LOGGER.isErrorEnabled()) {
                LOGGER.error("convert: ", e);
            }

            return ConsumeStatus.RECONSUME_LATER;
        }

        if (null == result) {
            return ConsumeStatus.RECONSUME_LATER;
        }

        try {
            if (result.getCode() == 1) {
                CountAwradmsgDetail entity = result.getCountAwardmsg();
                service.save(message.getMsgId(), entity);
            } else {
                CountTicketAdvertiser entity = result.getCountCertifiateAdshop();
                service.save(message.getMsgId(), entity);
            }

            return ConsumeStatus.CONSUME_SUCCESS;
        } catch (Exception e) {
            if (LOGGER.isErrorEnabled()) {
                LOGGER.error("consume: ", e);
            }
        }
        return ConsumeStatus.RECONSUME_LATER;
    }
}
