package com.bxm.datapark.service.service;

import com.bxm.datapark.constant.MongoConstant;
import com.bxm.datapark.dal.mongo.dao.AwardDetailDao;
import com.bxm.datapark.dal.mongo.ro.CommonRo;
import com.bxm.datapark.dal.mongo.vo.AwardDetail;
import com.bxm.datapark.enums.SumType;
import com.bxm.datapark.service.util.MongoSomeUtils;
import com.bxm.util.StringUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.mongodb.core.aggregation.Aggregation;
import org.springframework.data.mongodb.core.aggregation.Fields;
import org.springframework.data.mongodb.core.aggregation.MatchOperation;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.stereotype.Service;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static com.bxm.datapark.service.util.MongoSomeUtils.concat;

/**
 * 原始数据服务层
 *
 * @author 董朱旭
 * @version BUILD1001
 * @fileName com.bxm.datapark.service.service.AwardDetailService.java
 * @CopyRright (c) 2017-bxm：杭州微财科技有限公司
 * @created 2018-01-08 10:12:00
 * @modifier 董朱旭
 * @updated 2018-01-08 10:12:00
 * @description
 */
@Service
public class AwardDetailService {
    /**
     * 最原始的数据
     */
    @Autowired
    private AwardDetailDao awardDetailDao;


    /**
     * 按照各个维度统计那个原始数据
     *
     * @param datetime 日期
     * @param hour     时间
     * @return java.util.List<com.bxm.datapark.dal.mongo.vo.AwardDetail>
     * @description
     * @author 董朱旭
     * @time 2018/1/8
     * @CopyRight 杭州微财网络科技有限公司
     */
    private List<AwardDetail> findAwardDetail(String datetime, Integer hour, Integer sumType, Integer modeltype) throws ParseException {
        //若日期为 null
        if (StringUtil.isEmpty(datetime) || modeltype == null || sumType == null) {
            return null;
        }

        //初始条件
        Criteria criteria = Criteria.where(MongoConstant.MODELTYPE).is(modeltype).and(MongoConstant.BUSINESS).regex(MongoConstant.AD_BUSINESS);

        //是否还有小时作为条件
        MatchOperation match = dealHour(datetime, hour, criteria);

        if (SumType.CERTIFICATE.getCode().equals(sumType)) {
            return awardDetailDao
                    .sum(MongoSomeUtils.getAwardDetailCollectionName(datetime), match, Aggregation.group(Fields.from(Fields.field(MongoConstant.PREID, MongoConstant.PREID)).and(Fields.field(MongoConstant.MODELTYPE, MongoConstant.MODELTYPE)))
                            .sum(MongoConstant.VAILD)
                            .as(MongoConstant.VAILD)
                            .count()
                            .as(MongoConstant.COUNT));
        }

        if (SumType.CERTIFICATE_APP.getCode().equals(sumType)) {
            //开始查询吧
            return awardDetailDao
                    .sum(MongoSomeUtils.getAwardDetailCollectionName(datetime), match, Aggregation.group(Fields.from(Fields.field(MongoConstant.PREID, MongoConstant.PREID))
                            .and(Fields.field(MongoConstant.APPKEY, MongoConstant.APPKEY)))
                            .sum(MongoConstant.VAILD)
                            .as(MongoConstant.VAILD)
                            .count()
                            .as(MongoConstant.COUNT));
        }
        if (SumType.CERTIFICATE_APP_BUSINESS.getCode().equals(sumType)) {
            //开始查询吧
            return awardDetailDao
                    .sum(MongoSomeUtils.getAwardDetailCollectionName(datetime), match, Aggregation.group(Fields.from(Fields.field(MongoConstant.PREID, MongoConstant.PREID))
                            .and(Fields.field(MongoConstant.APPKEY, MongoConstant.APPKEY))
                            .and(Fields.field(MongoConstant.BUSINESS, MongoConstant.BUSINESS)))
                            .sum(MongoConstant.VAILD)
                            .as(MongoConstant.VAILD)
                            .count()
                            .as(MongoConstant.COUNT));
        }


        //其他情况先不管
        return null;
    }

    /**
     * 处理小时条件
     *
     * @param datetime
     * @param hour
     * @param criteria
     * @return org.springframework.data.mongodb.core.aggregation.MatchOperation
     * @description
     * @author 董朱旭
     * @time 2018/1/8
     * @CopyRight 杭州微财网络科技有限公司
     */
    private MatchOperation dealHour(String datetime, Integer hour, Criteria criteria) throws ParseException {
        if (hour != null) {
            //时间格式
            SimpleDateFormat sdf = new SimpleDateFormat(MongoConstant.DATE_FOMAT_DEFAULT);

            if (MongoConstant.TWENTY_THREE.equals(hour)) {
                criteria
                        .andOperator(Criteria.where(MongoConstant.CLICKTIME)
                                .gte(sdf.parse(MongoSomeUtils.getTime(datetime, hour)))
                                .lt(sdf.parse(MongoSomeUtils.getTime(MongoSomeUtils.getNextDate(datetime), MongoConstant.ZERO))));
            } else {
                //查询条件
                criteria
                        .andOperator(Criteria.where(MongoConstant.CLICKTIME)
                                .gte(sdf.parse(MongoSomeUtils.getTime(datetime, hour)))
                                .lt(sdf.parse(MongoSomeUtils.getTime(datetime, hour + MongoConstant.ONE))));
            }


        }
        //返回结果
        return Aggregation.match(criteria);
    }


    /**
     * 按礼券维度
     *
     * @param datetime
     * @param hour
     * @return java.util.List<com.bxm.datapark.dal.mongo.ro.CommonRo>
     * @author 董朱旭
     * @time 2018/1/9
     * @CopyRight 杭州微财网络科技有限公司
     */
    public List<CommonRo> taskForCertificate(String datetime, Integer hour) throws ParseException {
        return getList(datetime, hour, SumType.CERTIFICATE.getCode());
    }

    /**
     * 开发者维度
     *
     * @param datetime
     * @param hour
     * @return java.util.List<com.bxm.datapark.dal.mongo.ro.CommonRo>
     * @author 董朱旭
     * @time 2018/1/9
     * @CopyRight 杭州微财网络科技有限公司
     */
    public List<CommonRo> taskForApp(String datetime, Integer hour) throws ParseException {
        return getList(datetime, hour, SumType.CERTIFICATE_APP.getCode());
    }

    /**
     * 按子链接维度
     *
     * @param datetime
     * @param hour
     * @return java.util.List<com.bxm.datapark.dal.mongo.ro.CommonRo>
     * @author 董朱旭
     * @time 2018/1/9
     * @CopyRight 杭州微财网络科技有限公司
     */
    public List<CommonRo> taskForBusiness(String datetime, Integer hour) throws ParseException {
        return getList(datetime, hour, SumType.CERTIFICATE_APP_BUSINESS.getCode());
    }

    /**
     * 具体操作，代码复用
     *
     * @param datetime
     * @param hour
     * @param sumType
     * @return java.util.List<com.bxm.datapark.dal.mongo.ro.CommonRo>
     * @author 董朱旭
     * @time 2018/1/9
     * @CopyRight 杭州微财网络科技有限公司
     */
    private List<CommonRo> getList(String datetime, Integer hour, Integer sumType) throws ParseException {
        //查询 send_pv
        List<AwardDetail> sendPvs = findAwardDetail(datetime, hour, sumType, MongoConstant.SEND_PV);
        List<AwardDetail> openPvs = findAwardDetail(datetime, hour, sumType, MongoConstant.OPEN_PV);
        List<AwardDetail> clickPvs = findAwardDetail(datetime, hour, sumType, MongoConstant.CLICK_PV);

        //将有效点击存储到 map 减少遍历次数
        Map<String, Integer> mapOpenPv = new HashMap<String, Integer>();
        Map<String, Integer> mapClickPv = new HashMap<String, Integer>();
        Map<String, Integer> mapAvailable = new HashMap<String, Integer>();
        //若是礼券维度
        if (SumType.CERTIFICATE.getCode().equals(sumType)) {
            //存在有效点击
            //爆券量
            if (openPvs != null && !openPvs.isEmpty()) {
                openPvs
                        .stream()
                        .filter(openPv -> openPv.getPreid() != null)
                        .forEach(openPv -> mapOpenPv.put(openPv.getPreid().toString(), openPv.getCount()));
            }
            //存在有效点击
            //点击量
            if (clickPvs != null && !clickPvs.isEmpty()) {
                clickPvs
                        .stream()
                        .filter(clickPv -> clickPv.getPreid() != null)
                        .forEach(clickPv -> {
                            mapClickPv.put(clickPv.getPreid().toString(), clickPv.getCount());
                            mapAvailable.put(clickPv.getPreid().toString(), clickPv.getVaild());
                        });
            }
        } else
            //若是开发者维度
            if (SumType.CERTIFICATE_APP.getCode().equals(sumType)) {
                //爆券量
                if (openPvs != null && !openPvs.isEmpty()) {
                    openPvs
                            .stream()
                            .filter(openPv -> openPv.getPreid() != null && openPv.getAppkey() != null)
                            .forEach(openPv -> mapOpenPv.put(concat(openPv.getPreid().toString(), openPv.getAppkey()), openPv.getCount()));
                }
                //存在有效点击
                //点击量
                if (clickPvs != null && !clickPvs.isEmpty()) {
                    clickPvs
                            .stream()
                            .filter(clickPv -> clickPv.getPreid() != null && clickPv.getAppkey() != null)
                            .forEach(clickPv -> {
                                mapClickPv.put(concat(clickPv.getPreid().toString(), clickPv.getAppkey()), clickPv.getCount());
                                mapAvailable.put(concat(clickPv.getPreid().toString(), clickPv.getAppkey()), clickPv.getVaild());
                            });
                }
            } else
                //若是子链接维度
                if (SumType.CERTIFICATE_APP_BUSINESS.getCode().equals(sumType)) {
                    //爆券量
                    if (openPvs != null && !openPvs.isEmpty()) {
                        openPvs
                                .stream()
                                .filter(openPv -> openPv.getPreid() != null && openPv.getAppkey() != null && openPv.getBusiness() != null)
                                .forEach(openPv -> mapOpenPv.put(concat(concat(openPv.getPreid().toString(), openPv.getAppkey()), openPv.getBusiness()), openPv.getCount()));
                    }
                    //存在有效点击
                    //点击量
                    if (clickPvs != null && !clickPvs.isEmpty()) {
                        clickPvs
                                .stream()
                                .filter(clickPv -> clickPv.getPreid() != null && clickPv.getAppkey() != null && clickPv.getBusiness() != null)
                                .forEach(clickPv -> {
                                    mapClickPv.put(concat(concat(clickPv.getPreid().toString(), clickPv.getAppkey()), clickPv.getBusiness()), clickPv.getCount());
                                    mapAvailable.put(concat(concat(clickPv.getPreid().toString(), clickPv.getAppkey()), clickPv.getBusiness()), clickPv.getVaild());
                                });
                    }

                }

        //处理渠道维度的数据
        return MongoSomeUtils.dealCommonRo(sendPvs, mapOpenPv, mapClickPv, mapAvailable, sumType);
    }

}
