package com.bxm.fossicker.activity.lottery.service.impl;

import com.alibaba.fastjson.JSONObject;
import com.bxm.fossicker.activity.domain.lottery.LotteryParticipantMapper;
import com.bxm.fossicker.activity.lottery.config.LotteryPhaseStatus;
import com.bxm.fossicker.activity.lottery.service.LotteryDrawService;
import com.bxm.fossicker.activity.lottery.service.LotteryParticipantService;
import com.bxm.fossicker.activity.lottery.service.LotteryPhaseService;
import com.bxm.fossicker.activity.model.dto.lottery.LotteryJoinTime;
import com.bxm.fossicker.activity.model.dto.lottery.LotteryParticipantDTO;
import com.bxm.fossicker.activity.model.dto.lottery.LotteryPhaseDetailDTO;
import com.bxm.fossicker.activity.model.dto.lottery.LotteryWinnerCalDTO;
import com.bxm.fossicker.activity.model.param.lottery.LotteryDrawParam;
import com.bxm.fossicker.activity.model.param.lottery.LotteryPhaseQueryParam;
import com.bxm.fossicker.activity.model.vo.lottery.LotteryPhaseBean;
import com.bxm.fossicker.activity.model.vo.lottery.LotteryVirtaulUserBean;
import com.bxm.fossicker.activity.service.config.ActivityProperties;
import com.bxm.fossicker.message.enums.NotifyEnum;
import com.bxm.fossicker.message.enums.NotifyParamEnum;
import com.bxm.fossicker.message.enums.PushStrategyEnum;
import com.bxm.fossicker.message.facade.PushFacadeService;
import com.bxm.fossicker.message.param.MixPushParam;
import com.bxm.fossicker.thirdpart.facade.service.WxMpMessageFacadeService;
import com.bxm.fossicker.user.facade.VirtualUserFacadeService;
import com.bxm.fossicker.user.model.entity.VirtualUserBean;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.SpringContextHolder;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.math.NumberUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;

@Service
@Slf4j
public class LotteryDrawServiceImpl implements LotteryDrawService, ApplicationContextAware {

    /**
     * 参与计算的用户数量
     */
    private final static int CAL_NUM = 50;

    /**
     * 计算过程需要显示毫秒，增加特定的日期格式化
     */
    private final static ThreadLocal<DateFormat> CUSTOM_FORMAT = ThreadLocal.withInitial(() ->
            new SimpleDateFormat("yyyy-MM-dd hh:mm:ss.SSS"));

    private LotteryPhaseService lotteryPhaseService;

    private final LotteryParticipantMapper lotteryParticipantMapper;

    private final VirtualUserFacadeService virtualUserFacadeService;

    private final ActivityProperties activityProperties;

    private LotteryParticipantService lotteryParticipantService;

    private ApplicationContext applicationContext;

    private WxMpMessageFacadeService wxMpMessageFacadeService;

    private final PushFacadeService pushFacadeService;

    @Autowired
    public LotteryDrawServiceImpl(LotteryPhaseService lotteryPhaseService,
                                  LotteryParticipantMapper lotteryParticipantMapper,
                                  VirtualUserFacadeService virtualUserFacadeService,
                                  ActivityProperties activityProperties,
                                  WxMpMessageFacadeService wxMpMessageFacadeService,
                                  PushFacadeService pushFacadeService) {
        this.lotteryPhaseService = lotteryPhaseService;
        this.lotteryParticipantMapper = lotteryParticipantMapper;
        this.virtualUserFacadeService = virtualUserFacadeService;
        this.activityProperties = activityProperties;
        this.wxMpMessageFacadeService = wxMpMessageFacadeService;

        this.pushFacadeService = pushFacadeService;
    }

    private LotteryParticipantService getLotteryParticipantService() {
        if (this.lotteryParticipantService == null) {
            this.lotteryParticipantService = applicationContext.getBean(LotteryParticipantService.class);
        }
        return lotteryParticipantService;
    }

    @Override
    public void doDraw(Long phaseId, String title) {
        LotteryPhaseDetailDTO phaseDetail = lotteryPhaseService.loadCache(phaseId);

        if (phaseDetail.getStatus() != LotteryPhaseStatus.GOING.getCode()) {
            log.warn("活动不处理进行中状态，不进行开奖处理，phase:{}", phaseDetail);
            return;
        }

        //获取最后50 - 1条记录，计算中奖编号
        LotteryPhaseQueryParam pageParam = getLotteryPhaseQueryParam(phaseId, CAL_NUM - 1);
        List<LotteryParticipantDTO> lastJoinUsers = lotteryParticipantMapper.getParticipantByPage(pageParam);

        //分页排序为逆序，取第一位
        LotteryVirtaulUserBean lastVirtual = getLastJoinVirtualUser(lastJoinUsers.get(0));

        lastJoinUsers.add(LotteryParticipantDTO.builder()
                .headImg(lastVirtual.getHeadImg())
                .joinTime(lastVirtual.getAddTime())
                .nickName(lastVirtual.getNickname())
                .userId(lastVirtual.getId())
                .build());

        LotteryParticipantDTO winner = cal(lastJoinUsers, phaseDetail, lastVirtual);

        if (null != winner) {
            LotteryPhaseBean modifyPhase = LotteryPhaseBean.builder()
                    .status(LotteryPhaseStatus.FINISH.getCode())
                    .winnerCode(winner.getCode())
                    .winnerHeadUrl(winner.getHeadImg())
                    .winnerId(winner.getUserId())
                    .winnerName(winner.getNickName())
                    //开奖时间与最后一位虚拟用户参与时间一致
                    .winTime(lastVirtual.getAddTime())
                    .id(phaseId)
                    .build();

            //关闭本期活动
            lotteryPhaseService.closePhase(modifyPhase);
        }

        //向正常用户推送公众号开奖通知
        SpringContextHolder.getBean(LotteryDrawService.class).pushWxMessage(phaseId, title, lastVirtual.getAddTime());

        log.debug("活动已开奖：{},中奖人：{}", phaseDetail, winner);
    }

    @Override
    @Async
    public void pushWxMessage(Long phaseId, String title, Date time) {
        //公众号推送-开奖推送（查询此活动中的非虚拟用户，推送告知开奖）简单做一下
        List<LotteryParticipantDTO> realUsers = lotteryParticipantMapper.getParticipant(phaseId, (byte) 0);
        if (CollectionUtils.isEmpty(realUsers)) {
            log.info("活动：[{}]无真实用户参与", phaseId);
            return;
        }
        log.debug("当前活动:[{}],真实用户：[{}]", phaseId, JSONObject.toJSON(realUsers));

        //用户以及对应的券数量
        Map<Long, Long> userCodeMap = realUsers.stream().collect(Collectors.groupingBy(LotteryParticipantDTO::getUserId,
                Collectors.counting()));

        String drawTime = DateUtils.formatDateTime(time);

        userCodeMap.forEach((userId, num) -> {
            pushFacadeService.push(MixPushParam.builder()
                    .notifyType(NotifyEnum.INDIANAPOLIS_DRAW_MSG)
                    .userId(userId)
                    .addStrategy(PushStrategyEnum.WECHAT)
                    .addExtend(NotifyParamEnum.TITLE, title)
                    .addExtend(NotifyParamEnum.TIME, drawTime)
                    .addExtend(NotifyParamEnum.NUM, String.valueOf(num))
                    .build());
        });
    }


    /**
     * 计算最后的中奖人员
     *
     * @param lastJoinUsers 最后50条参与用户
     */
    private LotteryParticipantDTO cal(List<LotteryParticipantDTO> lastJoinUsers,
                                      LotteryPhaseDetailDTO phaseDetail,
                                      LotteryVirtaulUserBean lastVirtual) {
        LotteryParticipantDTO res = null;

        //获取所有参与的虚拟用户列表
        List<LotteryParticipantDTO> virtualUsers = lotteryParticipantMapper.getVitrualUser(phaseDetail.getPhaseId());
        virtualUsers.forEach(user -> user.setCodeIndex(restore(user.getCode())));

        //计算最后50用户参与时间的汇总
        long sum = lastJoinUsers.stream().mapToLong(user -> convert(user.getJoinTime())).sum();

        //中奖人员的索引编码，根据编码获取最近的中奖虚拟用户
        int winnerIndex = (int) (sum % phaseDetail.getConditionNum());

        int fixMillSeconds = 0;

        //正向查找
        for (LotteryParticipantDTO virtualUser : virtualUsers) {
            if (virtualUser.getCodeIndex() >= winnerIndex) {
                fixMillSeconds = virtualUser.getCodeIndex() - winnerIndex;
                res = virtualUser;
                break;
            }
        }

        //未匹配时，进行逆向查找
        if (res == null) {
            virtualUsers = virtualUsers.stream().sorted(Comparator.comparing(LotteryParticipantDTO::getCodeIndex).reversed())
                    .collect(Collectors.toList());

            for (LotteryParticipantDTO virtualUser : virtualUsers) {
                if (virtualUser.getCodeIndex() < winnerIndex) {
                    fixMillSeconds = virtualUser.getCodeIndex() - winnerIndex;
                    res = virtualUser;
                    break;
                }
            }
            log.debug("逆序查找虚拟中奖用户，修正毫秒数：{},中奖人：{}", fixMillSeconds, res);
        } else {
            log.debug("正序查找虚拟中奖用户，修正毫秒数：{},中奖人：{}", fixMillSeconds, res);
        }

        if (res == null) {
            log.error("开奖失败，未查找到符合条件的虚拟用户,活动期数：{}", phaseDetail);
            return null;
        }

        //修改最后一位虚拟用户的毫秒值
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(lastVirtual.getAddTime());

        calendar.add(Calendar.MILLISECOND, fixMillSeconds);
        lastVirtual.setAddTime(calendar.getTime());

        log.debug("开奖参数，总数：{}，预计中奖编号：{}，参加活动人数：{}，修正毫秒数：{}，修正后中奖人员：{},修正后参与时间：{}",
                sum,
                winnerIndex,
                phaseDetail.getParticipantNum(),
                fixMillSeconds,
                res.getCodeIndex(),
                CUSTOM_FORMAT.get().format(lastVirtual.getAddTime()));

        LotteryDrawParam joinParam = LotteryDrawParam.builder()
                .userId(lastVirtual.getId())
                .phaseId(phaseDetail.getPhaseId())
                .virtual(true)
                .lastVirtual(true)
                .virtaulUser(lastVirtual)
                .build();

        //将修正值添加到最后一位虚拟用户的参与时间上，保存到参与记录
        getLotteryParticipantService().addParticipant(joinParam);

        return res;
    }

    /**
     * 将奖券号码还原为参与活动的顺序编码
     *
     * @param code 奖券号码
     * @return 奖券顺序码
     */
    private int restore(String code) {
        int codeValue = NumberUtils.toInt(code);
        return codeValue - activityProperties.getCodePrefix();
    }

    /**
     * 创建最后一位参与的虚拟用户，用于使得计算公式正确并让虚拟用户中奖
     *
     * @return 虚拟用户信息
     */
    private LotteryVirtaulUserBean getLastJoinVirtualUser(LotteryParticipantDTO lastUser) {
        List<VirtualUserBean> virtualUsers = virtualUserFacadeService.listVirtualUser(1);
        VirtualUserBean virtualUser = virtualUsers.get(0);

        Calendar calendar = Calendar.getInstance();
        calendar.setTime(lastUser.getJoinTime());

        calendar.add(Calendar.SECOND, 1);
        //毫秒设置为0
        calendar.set(Calendar.MILLISECOND, 0);

        return LotteryVirtaulUserBean.builder()
                .id(virtualUser.getId())
                .sex(virtualUser.getSex())
                .nickname(virtualUser.getNickname())
                .headImg(virtualUser.getHeadImg())
                .addTime(calendar.getTime())
                .build();
    }

    @Override
    public LotteryWinnerCalDTO getCalProcess(Long phaseId) {
        LotteryPhaseDetailDTO detail = lotteryPhaseService.loadCache(phaseId);

        if (detail.getStatus() != LotteryPhaseStatus.FINISH.getCode()) {
            log.error("活动[{}]尚未结束，调动了查看计算过程的方法", phaseId);
            return LotteryWinnerCalDTO.builder().build();
        }

        //获取最后50条参与人数
        List<LotteryJoinTime> joinTimeList = getLastJoinParticipant(phaseId);

        return LotteryWinnerCalDTO.builder()
                .winnerCode(detail.getWinnerCode())
                .participantNum(detail.getParticipantNum())
                .joinTimeList(joinTimeList)
                .lastCount(sumJoinTime(joinTimeList))
                .build();
    }


    @SuppressWarnings("CodeBlock2Expr")
    private List<LotteryJoinTime> getLastJoinParticipant(Long phaseId) {
        LotteryPhaseQueryParam pageParam = getLotteryPhaseQueryParam(phaseId, CAL_NUM);
        List<LotteryParticipantDTO> lastJoinUsers = lotteryParticipantMapper.getParticipantByPage(pageParam);

        return lastJoinUsers.stream().map(item -> {
            return LotteryJoinTime.builder()
                    .joinTime(CUSTOM_FORMAT.get().format(item.getJoinTime()))
                    .convertTime(convert(item.getJoinTime()))
                    .build();
        }).collect(Collectors.toList());
    }

    private LotteryPhaseQueryParam getLotteryPhaseQueryParam(Long phaseId, int userNum) {
        LotteryPhaseQueryParam pageParam = new LotteryPhaseQueryParam();
        pageParam.setPageSize(userNum);
        pageParam.setPhaseId(phaseId);
        return pageParam;
    }

    private Long sumJoinTime(List<LotteryJoinTime> joinTimeList) {
        return joinTimeList.stream().mapToLong(LotteryJoinTime::getConvertTime).sum();
    }

    /**
     * 将日期转换为整形，如：2020-02-10 10:01:11.123
     * 转换结果为：100111123
     * 取时分秒毫秒组成字符串，然后转换为Long型
     *
     * @param date 原始日期数据
     * @return 转化结果
     */
    private Long convert(Date date) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(date);

        int hour = calendar.get(Calendar.HOUR);
        int minute = calendar.get(Calendar.MINUTE);
        int second = calendar.get(Calendar.SECOND);
        int millisecond = calendar.get(Calendar.MILLISECOND);

        String finalNumStr = String.valueOf(hour) +
                String.format("%02d", minute) +
                String.format("%02d", second) +
                String.format("%03d", millisecond);
        return NumberUtils.toLong(finalNumStr);
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }
}
