package com.bxm.fossicker.activity.lottery.service.impl;

import com.bxm.fossicker.activity.domain.lottery.LotteryMapper;
import com.bxm.fossicker.activity.domain.lottery.LotteryParticipantMapper;
import com.bxm.fossicker.activity.domain.lottery.LotteryPhaseMapper;
import com.bxm.fossicker.activity.facade.model.LotteryDTO;
import com.bxm.fossicker.activity.lottery.config.LotteryPhaseStatus;
import com.bxm.fossicker.activity.lottery.service.LotteryParticipantService;
import com.bxm.fossicker.activity.lottery.service.LotteryPhaseService;
import com.bxm.fossicker.activity.model.dto.lottery.LotteryCarouselDTO;
import com.bxm.fossicker.activity.model.dto.lottery.LotteryCompleteDTO;
import com.bxm.fossicker.activity.model.dto.lottery.LotteryParticipantDTO;
import com.bxm.fossicker.activity.model.dto.lottery.LotteryPhaseDetailDTO;
import com.bxm.fossicker.activity.model.param.lottery.LotteryPhaseQueryParam;
import com.bxm.fossicker.activity.model.param.lottery.LotteryQueryParam;
import com.bxm.fossicker.activity.model.vo.lottery.LotteryBean;
import com.bxm.fossicker.activity.model.vo.lottery.LotteryPhaseBean;
import com.bxm.fossicker.vo.PageWarper;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisListAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.vo.Message;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.Date;
import java.util.List;
import java.util.Map;

import static com.bxm.fossicker.activity.constants.ActivityRedisKeyConstant.LOTTERY_CAROUSEL_KEY;
import static com.bxm.fossicker.activity.constants.ActivityRedisKeyConstant.LOTTERY_PHASE_KEY;

@Service
public class LotteryPhaseServiceImpl extends BaseService implements LotteryPhaseService {

    private final LotteryPhaseMapper lotteryPhaseMapper;

    private final LotteryMapper lotteryMapper;

    private final LotteryParticipantMapper lotteryParticipantMapper;

    private final RedisStringAdapter redisStringAdapter;

    private final RedisListAdapter redisListAdapter;

    private LotteryParticipantService lotteryParticipantService;

    @Autowired
    public LotteryPhaseServiceImpl(LotteryPhaseMapper lotteryPhaseMapper,
                                   LotteryMapper lotteryMapper,
                                   LotteryParticipantMapper lotteryParticipantMapper,
                                   RedisStringAdapter redisStringAdapter,
                                   RedisListAdapter redisListAdapter) {
        this.lotteryPhaseMapper = lotteryPhaseMapper;
        this.lotteryMapper = lotteryMapper;
        this.lotteryParticipantMapper = lotteryParticipantMapper;
        this.redisStringAdapter = redisStringAdapter;
        this.redisListAdapter = redisListAdapter;
    }

    @Override
    public void createPhase(LotteryBean lottery) {
        //保存期数信息
        LotteryPhaseBean newPhase = LotteryPhaseBean.builder()
                .id(nextId())
                .status(LotteryPhaseStatus.GOING.getCode())
                .conditionNum(lottery.getConditionNum())
                .participantNum(0)
                .createTime(new Date())
                .phaseNum(String.valueOf(lottery.getPhaseCount() + 1))
                .lotteryId(lottery.getId())
                .build();

        lotteryPhaseMapper.insert(newPhase);
        //变更历史活动的最新一期数据，用于客户端引导最新一期活动
        changeLastPhase(lottery.getId(), newPhase.getId());
    }

    @Override
    public void resumePhase(LotteryBean lottery) {
        //获取最后一期活动
        LotteryPhaseBean lastPhase = lotteryPhaseMapper.getLastPhase(lottery.getId(), null);

        //已完成状态的则开启新一期
        if (null == lastPhase || lastPhase.getStatus() == LotteryPhaseStatus.FINISH.getCode()) {
            createPhase(lottery);
            return;
        }
        //进行中的
        if (lastPhase.getStatus() == LotteryPhaseStatus.GOING.getCode()) {
            return;
        }

        //将活动最后一期更新为激活状态
        lastPhase.setStatus(LotteryPhaseStatus.GOING.getCode());
        lotteryPhaseMapper.updateByPrimaryKeySelective(lastPhase);
    }

    @Override
    public void closePhase(LotteryPhaseBean entity) {
        //更新活动期状态
        lotteryPhaseMapper.updateByPrimaryKeySelective(entity);

        //清理缓存
        cleanCache(entity.getId());

        //开启新一期活动
        LotteryPhaseBean phaseInfo = lotteryPhaseMapper.selectByPrimaryKey(entity.getId());

        //活动期数增加
        lotteryMapper.addNum(phaseInfo.getLotteryId());

        LotteryBean lotteryBean = lotteryMapper.selectByPrimaryKey(phaseInfo.getLotteryId());

        changeLastPhase(phaseInfo.getLotteryId(), null);

        createPhase(lotteryBean);

        //保存中奖信息，用于轮播显示
        redisListAdapter.leftPush(LOTTERY_CAROUSEL_KEY, LotteryCarouselDTO.builder()
                .headImg(entity.getWinnerHeadUrl())
                .lotteryTitle(lotteryBean.getTitle())
                .nickName(entity.getWinnerName())
                .build());
    }

    @Override
    public List<LotteryCarouselDTO> getCarousel() {
        List<LotteryCarouselDTO> carouselList = redisListAdapter.range(LOTTERY_CAROUSEL_KEY,
                0,
                5,
                LotteryCarouselDTO.class);

        if (carouselList.size() < 5) {
            return Lists.newArrayList();
        }

        return carouselList;
    }

    @Override
    public Message modify(LotteryBean lottery) {
        //获取正在进行中的最后一期活动
        LotteryPhaseBean lastPhase = lotteryPhaseMapper.getLastPhase(lottery.getId(),
                LotteryPhaseStatus.GOING.getCode());
        //人数大于当前开奖需要的人数，则更新
        Message message;
        if (null != lastPhase && lottery.getConditionNum() > lastPhase.getConditionNum()) {
            logger.debug("活动信息变更，原始开奖人数：{},现开奖人数：{}",
                    lastPhase.getConditionNum(),
                    lottery.getConditionNum());

            lastPhase.setConditionNum(lottery.getConditionNum());
            message = Message.build(lotteryPhaseMapper.updateByPrimaryKeySelective(lastPhase));

        } else {
            message = Message.build();
            logger.info("活动开奖人数未变化，不做处理,参数：{}", lottery);
        }
        return message;
    }

    @Override
    public Message cancel(LotteryBean lottery) {
        //获取最后一期活动
        LotteryPhaseBean lastPhase = lotteryPhaseMapper.getLastPhase(lottery.getId(), LotteryPhaseStatus.GOING.getCode());
        //判断最后一期活动中是否存在真实参与用户
        if (null != lastPhase) {
            int realUserCount = lotteryParticipantMapper.getRealUserCount(lastPhase.getId());

            if (realUserCount == 0) {
                //将活动设置为取消状态
                LotteryPhaseBean modifyEntity = LotteryPhaseBean.builder()
                        .id(lastPhase.getId())
                        .status(LotteryPhaseStatus.CANCEL.getCode())
                        .build();
                lotteryPhaseMapper.updateByPrimaryKeySelective(modifyEntity);
            } else {
                logger.info("活动存在真实参与用户，不进行取消,phaseId：{}", lastPhase.getId());
            }
        }
        return Message.build();
    }

    @Override
    public void changeLastPhase(Long lotteryId, Long phaseId) {
        Preconditions.checkArgument(null != lotteryId);

        lotteryPhaseMapper.setLastPhaseId(lotteryId, phaseId);

        List<Long> otherPhaseIds = lotteryPhaseMapper.getAllPhaseByLottery(lotteryId, phaseId);

        if (!CollectionUtils.isEmpty(otherPhaseIds)) {
            List<KeyGenerator> removeKeys = Lists.newArrayList();
            for (Long otherPhaseId : otherPhaseIds) {
                removeKeys.add(buildKey(otherPhaseId));
            }

            redisStringAdapter.remove(removeKeys);
        }

    }

    @Override
    public LotteryPhaseDetailDTO loadCache(Long phaseId) {
        KeyGenerator key = buildKey(phaseId);
        LotteryPhaseDetailDTO res = redisStringAdapter.get(key, LotteryPhaseDetailDTO.class);

        if (null == res) {
            res = lotteryPhaseMapper.getDetail(phaseId);

            if (res == null) {
                res = LotteryPhaseDetailDTO.builder().build();
                res.setPhaseId(phaseId);
                redisStringAdapter.set(key, res, 60);
            } else {
                redisStringAdapter.set(key, res);
            }
        }

        return res;
    }

    @Override
    public void cleanCache(Long phaseId) {
        KeyGenerator key = buildKey(phaseId);
        redisStringAdapter.remove(key);
    }

    private LotteryParticipantService getLotteryParticipantService() {
        if (this.lotteryParticipantService == null) {
            this.lotteryParticipantService = SpringContextHolder.getBean(LotteryParticipantService.class);
        }
        return this.lotteryParticipantService;
    }

    @Override
    public PageWarper<LotteryDTO> getPhaseList(LotteryQueryParam param) {
        List<LotteryDTO> result = lotteryPhaseMapper.getPhaseByPage(param);

        Map<String, Integer> userJoinHistories = getLotteryParticipantService().loadJoinHistoryMap(param.getUserId(), null);

        for (LotteryDTO item : result) {
            Integer joinTimes = userJoinHistories.get(String.valueOf(item.getPhaseId()));
            if (null != joinTimes && joinTimes > 0) {
                item.setTimes(joinTimes);
            }
        }

        return new PageWarper<>(result);
    }

    @Override
    public PageWarper<LotteryCompleteDTO> getCompletePhaseList(LotteryPhaseQueryParam param) {
        List<LotteryCompleteDTO> result = lotteryPhaseMapper.getPhaseCompleteByPage(param);

        Map<String, Integer> userJoinHistories = getLotteryParticipantService().loadJoinHistoryMap(param.getUserId(), null);

        for (LotteryDTO item : result) {
            Integer joinTimes = userJoinHistories.get(String.valueOf(item.getPhaseId()));
            if (null != joinTimes && joinTimes > 0) {
                item.setTimes(joinTimes);
            }
        }

        return new PageWarper<>(result);
    }

    @Override
    public LotteryPhaseDetailDTO getDetail(LotteryPhaseQueryParam param) {
        //活动基础信息
        LotteryPhaseDetailDTO detailDTO = loadCache(param.getPhaseId());

        //我的参与奖券号码
        detailDTO.setMyCodes(lotteryParticipantMapper.getJoinHistoryCodes(param.getPhaseId(), param.getUserId()));
        if(null != detailDTO.getMyCodes() && detailDTO.getMyCodes().size() > 0){
            detailDTO.setTimes(detailDTO.getMyCodes().size());
        }

        //第一页参与人员
        LotteryPhaseQueryParam pageParam = new LotteryPhaseQueryParam();
        pageParam.setPhaseId(param.getPhaseId());
        pageParam.setUserId(param.getUserId());
        PageWarper<LotteryParticipantDTO> participantList = getLotteryParticipantService().getParticipantList(pageParam);

        detailDTO.setFirstPageParticipant(participantList);

        return detailDTO;
    }

    private KeyGenerator buildKey(Long phaseId) {
        return LOTTERY_PHASE_KEY.copy().appendKey(phaseId);
    }
}
