package com.bxm.fossicker.activity.service.advert.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.bxm.fossicker.activity.config.ExternalAdvertConfig;
import com.bxm.fossicker.activity.constants.ActivityRedisKeyConstant;
import com.bxm.fossicker.activity.constants.AdvertPrecondition;
import com.bxm.fossicker.activity.domain.ActivityAdvertMapper;
import com.bxm.fossicker.activity.model.dto.AdvertDto;
import com.bxm.fossicker.activity.model.entry.AdvertEntry;
import com.bxm.fossicker.activity.model.enums.AdvertPositionEnum;
import com.bxm.fossicker.activity.model.param.AdvertParam;
import com.bxm.fossicker.activity.model.param.AdvertRelationParam;
import com.bxm.fossicker.activity.model.vo.AppInfoForSwlh;
import com.bxm.fossicker.activity.service.advert.AdvertService;
import com.bxm.fossicker.activity.service.advert.AdvertisementFilterChainService;
import com.bxm.fossicker.activity.service.config.AdverProperties;
import com.bxm.fossicker.base.facade.AppVersionFacadeService;
import com.bxm.fossicker.base.facade.model.EquipmentDTO;
import com.bxm.fossicker.base.facade.param.AppVersionParam;
import com.bxm.fossicker.base.facade.service.EquipmentFacadeService;
import com.bxm.fossicker.base.facade.vo.AppChannelVersionVO;
import com.bxm.fossicker.commodity.common.utils.OkHttpUtils;
import com.bxm.fossicker.commodity.facade.CommodityFacadeService;
import com.bxm.fossicker.enums.PlatformEnum;
import com.bxm.fossicker.user.facade.UserInfoFacadeService;
import com.bxm.fossicker.user.facade.dto.UserInfoDto;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import com.taobao.api.request.TbkActivitylinkGetRequest;
import com.taobao.api.response.TbkActivitylinkGetResponse;
import lombok.extern.log4j.Log4j2;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.io.IOException;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 广告实现
 **/
@SuppressWarnings({"deprecation", "AliDeprecation"})
@Log4j2
@Service
public class AdvertServiceImpl implements AdvertService {

    private final ActivityAdvertMapper advertMapper;

    private final RedisHashMapAdapter redisHashMapAdapter;

    private final AppVersionFacadeService appVersionFacadeService;

    private final AdvertisementFilterChainService advertisementFilterChainService;

    private final UserInfoFacadeService userInfoFacadeService;

    private final ExternalAdvertConfig externalAdvertConfig;

    private final UserInfoFacadeService userInfoService;

    private final RedisStringAdapter redisStringAdapter;

    private final EquipmentFacadeService equipmentFacadeService;

    private final CommodityFacadeService commodityFacadeService;

    private final AdverProperties adverProperties;

    private TypeReference<List<AdvertDto>> typeReference = new TypeReference<List<AdvertDto>>() {
    };

    private LoadingCache<Integer, List<AdvertDto>> postionCache;

    private LoadingCache<Long, List<AdvertDto>> detailCache;

    public AdvertServiceImpl(ActivityAdvertMapper advertMapper, RedisHashMapAdapter redisHashMapAdapter,
                             AppVersionFacadeService appVersionFacadeService,
                             AdvertisementFilterChainService advertisementFilterChainService,
                             UserInfoFacadeService userInfoFacadeService, ExternalAdvertConfig externalAdvertConfig,
                             UserInfoFacadeService userInfoService, RedisStringAdapter redisStringAdapter,
                             EquipmentFacadeService equipmentFacadeService, CommodityFacadeService commodityFacadeService,
                             AdverProperties adverProperties) {

        this.advertMapper = advertMapper;
        this.redisHashMapAdapter = redisHashMapAdapter;
        this.appVersionFacadeService = appVersionFacadeService;
        this.advertisementFilterChainService = advertisementFilterChainService;
        this.userInfoFacadeService = userInfoFacadeService;
        this.externalAdvertConfig = externalAdvertConfig;
        this.userInfoService = userInfoService;
        this.redisStringAdapter = redisStringAdapter;
        this.equipmentFacadeService = equipmentFacadeService;
        this.commodityFacadeService = commodityFacadeService;
        this.adverProperties = adverProperties;
    }

    /**
     * 根据广告位获取广告列表，放入redis缓存
     *
     * @param position 广告位type
     * @return 广告列表
     */
    private List<AdvertDto> getByPosition(Integer position) {
        return loadFromDb(position);
    }

    private List<AdvertDto> loadFromDb(Integer position) {
        List<AdvertDto> advertDtoList = Lists.newArrayList();
        List<AdvertEntry> advertEntries = advertMapper.listByPositionType(position);

        if (advertEntries.size() > 0) {
            List<Long> relationIds = advertEntries.stream().map(AdvertEntry::getRelationId).collect(Collectors.toList());
            List<AdvertEntry> relationInfos = advertMapper.listByRelationId(relationIds);

            advertEntries.forEach(advert -> relationInfos.forEach(relation -> {
                if (Objects.equals(advert.getRelationId(), relation.getRelationId())
                        && advert.getMaterialId() == null) {
                    BeanUtils.copyProperties(relation, advert);
                }
            }));

            advertDtoList = advertEntries.stream()
                    .map(this::convert)
                    .filter(advert -> advert.getMaterialId() != null)
                    .collect(Collectors.toList());
        }

        return advertDtoList;
    }

    /**
     * 从缓存中加载广告
     *
     * @param position 广告位type
     * @return 广告列表
     */
    private List<AdvertDto> loadCache(Integer position) {
        if (null == postionCache) {
            postionCache = CacheBuilder.newBuilder()
                    .expireAfterWrite(adverProperties.getMemoryCacheExpireMinute(), TimeUnit.MINUTES)
                    .maximumSize(2000L)
                    .build(new CacheLoader<Integer, List<AdvertDto>>() {
                        @Override
                        public List<AdvertDto> load(Integer key) {
                            return getByPosition(key);
                        }
                    });
        }

        List<AdvertDto> advertDtos = postionCache.getUnchecked(position);
        if (CollectionUtils.isEmpty(advertDtos)) {
            return Lists.newArrayList();
        }

        return advertDtos.stream().map(p -> {
            AdvertDto advertDto = new AdvertDto();
            BeanUtils.copyProperties(p, advertDto);
            return advertDto;
        }).collect(Collectors.toList());
    }

    @Override
    public List<AdvertDto> list(AdvertParam param) {
        // 获取广告、过滤
        List<AdvertDto> advertDtos = filter(loadCache(param.getPosition()), param);

        if (!CollectionUtils.isEmpty(advertDtos)) {
            attach(param, advertDtos);
        }
        return advertDtos;
    }

    @Override
    public List<AdvertDto> getRelationDetail(AdvertRelationParam param) {
        if (null == detailCache) {
            detailCache = CacheBuilder.newBuilder()
                    .expireAfterWrite(adverProperties.getMemoryCacheExpireMinute(), TimeUnit.MINUTES)
                    .maximumSize(1000L)
                    .build(new CacheLoader<Long, List<AdvertDto>>() {
                        @Override
                        public List<AdvertDto> load(Long key) {
                            return advertMapper.listByRelationId(ImmutableList.of(key))
                                    .stream()
                                    .map(item -> convert(item))
                                    .collect(Collectors.toList());
                        }
                    });
        }
        return detailCache.getUnchecked(param.getRelationId());
    }


    private void attach(AdvertParam param, List<AdvertDto> advertDtos) {
        // 用户的淘宝客 关系id
        String tbkRelationId = userInfoService.getTbkRelationId(param.getUserId());

        advertDtos.forEach(p -> {
            // 表示该广告位要跳转淘宝联盟官方活动
            if (Objects.nonNull(p) && StringUtils.isNotBlank(tbkRelationId)
                    && StringUtils.isNotBlank(p.getAdzoneId()) && StringUtils.isNotBlank(p.getPromotionSceneId())) {

                Long adzoneId = null, promotionSceneId = null;
                try {
                    adzoneId = Long.valueOf(p.getAdzoneId());
                    promotionSceneId = Long.parseLong(p.getPromotionSceneId());
                } catch (Exception e) {
                    log.warn("类型转换失败，materialId: {} 的广告位配置有问题", p.getMaterialId());
                }

                if (Objects.nonNull(adzoneId) && Objects.nonNull(promotionSceneId)) {
                    // 获取淘宝联盟官方活动链接
                    String taoBaoUnionUrl = getTaoBaoUnionUrl(adzoneId, promotionSceneId, tbkRelationId);

                    p.setUrl(StringUtils.isNotBlank(taoBaoUnionUrl)
                            ? taoBaoUnionUrl.replaceAll("https", "qtj://profession/openApp?url=tbopen")
                            : "");
                }

            }
        });
    }

    /**
     * 过滤广告内容（审核控制、授权条件、字段填充）
     *
     * @param advertDtoList 广告列表
     * @param param         广告查询入参
     * @return 过滤后的广告列表
     */
    private List<AdvertDto> filter(List<AdvertDto> advertDtoList, AdvertParam param) {
        if (CollectionUtils.isEmpty(advertDtoList)) {
            return Lists.newArrayList();
        }
        // 过滤掉不需要显示的
        advertisementFilterChainService.filter(advertDtoList, param);

        //进行特殊列表过滤
        List<AdvertDto> list = doResult(advertDtoList, param);

        //对是否展示的列表做处理 version 1.4
        if (!CollectionUtils.isEmpty(list)) {
            handlePreconditions(list, param);
        }
        return list;
    }

    private void listForPlatform(List<AdvertDto> listSource, AdvertParam param) {
        //安卓
        if (param.getPlatform() == PlatformEnum.ANDROID.getCode()) {
            listSource.removeIf(advertDto -> advertDto.getPlatform() == PlatformEnum.ANDROID.getCode());
        }

        //ios
        if (param.getPlatform() == PlatformEnum.IOS.getCode()) {
            listSource.removeIf(advertDto -> advertDto.getPlatform() == PlatformEnum.IOS.getCode());
        }
    }

    private List<AdvertDto> doResult(List<AdvertDto> listSource, AdvertParam param) {
        //平台过滤
        listForPlatform(listSource, param);

        AdvertPositionEnum advertPosition = AdvertPositionEnum.getByPosition(param.getPosition());
        if (advertPosition == null) {
            return listSource;
        }

        switch (advertPosition) {
            //淘金谷逻辑
            case TJG_BANNER:
                //1.3.0 ios不下发淘金谷
                if (!externalAdvertConfig.isExternalTjgFlag()) {
                    log.info("关闭淘金谷");
                    return null;
                }
                String tjgTotalUrl = externalAdvertConfig.getTjgTotalUrl();
                listSource.forEach(advertDto ->
                        advertDto.setTotalUrl(tjgTotalUrl)
                );
                break;
            //试玩零花 bannel
            case SWLH_BANNER:
                String androidImei = getAndroidImei(param.getDevcId());
                if (!externalAdvertConfig.isExternalSwlhFlag() || androidImei == null) {
                    log.info("关闭试玩零花bannel");
                    return null;
                }
                UserInfoDto userInfo = userInfoService.getUserById(param.getUserId());
                if (userInfo == null) {
                    return null;
                }
                String swlhTotalUrl = externalAdvertConfig.getSwlhTotalUrl().replace("{channel}", externalAdvertConfig
                        .getChannel())
                        .replace("{uid}",
                                userInfo.getId().toString()).replace("{nickname}", userInfo.getNickName())
                        .replace("{imei}",
                                androidImei).replace("{headImgUrl}", userInfo.getHeadImg());
                listSource.forEach(advertDto -> advertDto.setTotalUrl(swlhTotalUrl));
                break;
            case SWLH_APP_LIST:
                String androidImei13 = getAndroidImei(param.getDevcId());
                if (!externalAdvertConfig.isExternalSwlhFlag() || androidImei13 == null) {
                    log.info("关闭试玩零花app列表");
                    return null;
                }
                listSource = getSwlhList(androidImei13, param.getUserId());
                break;
            default:
        }
        return listSource;
    }

    /**
     * 对广告位的前置条件进行处理
     */
    private void handlePreconditions(List<AdvertDto> list, AdvertParam param) {
        // 获取最新版本信息
        AppChannelVersionVO appChannelVersionVO = appVersionFacadeService.getNewversion(new AppVersionParam(param.getPlatform(),
                param.getChnl()));
        // 是否可以更新
        boolean canUpdate = null != appChannelVersionVO && Objects.nonNull(appChannelVersionVO.getVersion())
                && com.bxm.newidea.component.tools.StringUtils.compareVersion
                (param.getCurVer(), appChannelVersionVO.getVersion()) < 0;

        // 老版本处理
        handlePrecondition(list, param, appChannelVersionVO, canUpdate);

        // 新版本处理 新版本的处理是下发全部的版本处理信息 让客户端自己判断
        list.forEach(p -> {
            if (!CollectionUtils.isEmpty(p.getPreconditions())) {
                p.setPreconditions(p.getPreconditions().stream()
                        .peek(p1 -> {
                            // 唯独对于版本更新的前置条件 需要下发新版本的下载链接
                            // 其实这一步都不需要，客户端本来就有这个功能了，直接可以去拿的
                            if (AdvertPrecondition.HAS_HIGHER_VERSION.getCode() == p1 && canUpdate) {
                                p.setDownloadUrl(appChannelVersionVO.getDownloadLink());
                            }
                        })
                        // 转换成枚举
                        .map(AdvertPrecondition::getByCode)
                        // 过滤空的
                        .filter(Objects::nonNull)
                        // 排序
                        .sorted(Comparator.comparingInt(AdvertPrecondition::getOrder))
                        // 收集成前置条件
                        .map(AdvertPrecondition::getCode)
                        .collect(Collectors.toList()));
            } else {
                // 如果新字段为空 老字段有值且不为0 则将老版本字段兼容到新版本上面
                if (Objects.nonNull(p.getPrecondition())
                        && p.getPrecondition() != 0) {
                    p.setPreconditions(Collections.singletonList(p.getPrecondition()));
                }
            }
        });
    }

    /**
     * 对广告位的前置条件进行处理
     */
    private void handlePrecondition(List<AdvertDto> list, AdvertParam param, AppChannelVersionVO appChannelVersionVO,
                                    boolean canUpdate) {

        // 查询当前用户是否已经绑定过淘宝号
        boolean hasBindtaobao = userInfoFacadeService.hasBindTaobao(param.getUserId());

        for (AdvertDto advertDto : list) {

            // 新字段如果有值 且有更新信息 则老字段下发版本更新信息
            if (!CollectionUtils.isEmpty(advertDto.getPreconditions()) && canUpdate) {
                advertDto.setPrecondition(AdvertPrecondition.HAS_HIGHER_VERSION.getCode());
                advertDto.setDownloadUrl(appChannelVersionVO != null ? appChannelVersionVO.getDownloadLink() : null);
                continue;
            }

            // 原有的老板本处理信息
            if (null != advertDto.getPrecondition()) {
                if (advertDto.getPrecondition() == AdvertPrecondition.HAS_HIGHER_VERSION.getCode()) {
                    if (null != appChannelVersionVO && Objects.nonNull(appChannelVersionVO.getVersion())
                            && com.bxm.newidea.component.tools.StringUtils.compareVersion
                            (param.getCurVer(), appChannelVersionVO.getVersion()) < 0) {
                        advertDto.setPrecondition(AdvertPrecondition.HAS_HIGHER_VERSION.getCode());
                        advertDto.setDownloadUrl(appChannelVersionVO.getDownloadLink());
                        continue;
                    }
                } else if (advertDto.getPrecondition() == AdvertPrecondition.BIND_TAOBAO.getCode()) {
                    advertDto.setPrecondition(hasBindtaobao ? 0 : 1);
                    continue;
                }
            }
            advertDto.setPrecondition(0);
        }
    }


    private List<AdvertDto> getSwlhList(String imei, Long userId) {
        List<AdvertDto> list;
        KeyGenerator wellChoseKey = ActivityRedisKeyConstant.ADVERD_SWLH_APP_INFO.copy();
        String swlhInfoByKey = redisStringAdapter.getString(wellChoseKey);
        if (StringUtils.isNotBlank(swlhInfoByKey)) {
            list = JSONArray.parseArray(swlhInfoByKey, AdvertDto.class);
        } else {
            list = convert(imei, userId);
            if (list.size() > 0) {
                //缓存五分钟
                redisStringAdapter.set(wellChoseKey, JSON.toJSONString(list), 60 * 5);
            }
        }
        return list;
    }

    private List<AdvertDto> convert(String imei, Long userId) {
        List<AdvertDto> resultList = new ArrayList<>();
        String resultStr = null;
        try {
            resultStr = OkHttpUtils.get(externalAdvertConfig.getSwlhApiUrl().replace("{imei}", imei).replace
                    ("{uid}", userId.toString()));
        } catch (IOException e) {
            log.error("试玩零花交互异常  ： {}", e);
        }
        UserInfoDto userInfo = userInfoService.getUserById(userId);
        if (StringUtils.isNotBlank(resultStr) && resultStr.startsWith("[{") && userInfo != null) {
            List<AppInfoForSwlh> swlhList = JSONArray.parseArray(resultStr, AppInfoForSwlh.class);
            for (AppInfoForSwlh appInfoForSwlh : swlhList) {
                AdvertDto advertDto = new AdvertDto();
                convert(advertDto, appInfoForSwlh, userInfo, imei);
                resultList.add(advertDto);
            }
        }
        return resultList;
    }

    private void convert(AdvertDto advertDto, AppInfoForSwlh appInfoForSwlh, UserInfoDto userInfo, String imei) {
        String totalUrl = externalAdvertConfig.getSwlhTotalUrl().replace("{channel}", externalAdvertConfig.getChannel())
                .replace("{uid}",
                        userInfo.getId().toString()).replace("{nickname}", userInfo.getNickName())
                .replace("{imei}",
                        imei).replace("{headImgUrl}", userInfo.getHeadImg());

        advertDto.setTitle(appInfoForSwlh.getAdname());
        advertDto.setSubTitle(appInfoForSwlh.getIntro());
        advertDto.setShowmoney(appInfoForSwlh.getShowmoney());
        advertDto.setImgUrl(appInfoForSwlh.getImgurl());
        advertDto.setUrl(totalUrl);
        advertDto.setTotalUrl(totalUrl);
    }

    private AdvertDto convert(AdvertEntry advertEntry) {
        AdvertDto advertDto = new AdvertDto();
        BeanUtils.copyProperties(advertEntry, advertDto);

        // 活动id
        advertDto.setPromotionSceneId(advertEntry.getUrl());

        // 多个前置条件
        List<Integer> preconditions = Lists.newArrayList();
        if (StringUtils.isNotBlank(advertEntry.getPreconditions())) {
            preconditions = Arrays.stream(advertEntry.getPreconditions().split(","))
                    .map(Integer::parseInt)
                    .collect(Collectors.toList());
        }
        advertDto.setPreconditions(preconditions);
        return advertDto;
    }

    private String getAndroidImei(String deviceId) {
        EquipmentDTO equipmentDTO = equipmentFacadeService.getByDeviceId(deviceId);
        if (equipmentDTO == null) {
            return null;
        }
        if (equipmentDTO.getImei() != null) {
            return equipmentDTO.getImei();
        } else if (equipmentDTO.getAndroidId() != null) {
            return equipmentDTO.getAndroidId();
        } else if (equipmentDTO.getAndroidUuid() != null) {
            return equipmentDTO.getAndroidUuid();
        } else {
            return null;
        }
    }

    /**
     * 处理淘宝联盟跳转链接
     *
     * @return url
     */
    private String getTaoBaoUnionUrl(Long adzoneId, Long promotionSceneId, String relationId) {
        TbkActivitylinkGetRequest req = new TbkActivitylinkGetRequest();
        // 推广位id
        req.setAdzoneId(adzoneId);
        // relation_id
        req.setRelationId(relationId);
        // 官方活动id
        req.setPromotionSceneId(promotionSceneId);

        TbkActivitylinkGetResponse taoBaoUnion = commodityFacadeService.getTaoBaoUnion(req);

        if (Objects.nonNull(taoBaoUnion) && StringUtils.isNotBlank(taoBaoUnion.getData())) {
            return taoBaoUnion.getData().replaceAll("\\\\", "");
        }

        return "";
    }

}
