package com.bxm.fossicker.activity.service.advert.impl;

import com.alibaba.fastjson.JSONArray;
import com.bxm.fossicker.activity.model.dto.AdvertStrategyConfigDTO;
import com.bxm.fossicker.activity.model.dto.AdvertStrategyDTO;
import com.bxm.fossicker.activity.service.advert.AdvertStrategyService;
import com.bxm.fossicker.activity.service.config.ActivityProperties;
import com.bxm.fossicker.vo.BasicParam;
import com.bxm.newidea.component.tools.StringUtils;
import com.google.common.collect.Maps;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Map;

@Service
public class AdvertStrategyServiceImpl implements AdvertStrategyService {

    private final ActivityProperties activityProperties;

    /**
     * 解析结果缓存
     */
    private Map<String, List<AdvertStrategyDTO>> cacheMap;

    /**
     * 最后一次解析时间
     */
    private long lastParseTime;

    /**
     * 默认匹配的广告获取策略
     */
    private List<AdvertStrategyDTO> defaultMatch;

    /**
     * 解析间隔
     */
    private static final long PARSE_INTERVAL = 5 * 60 * 1000;

    /**
     * 默认的广告策略名称，配置到version字段上
     */
    private static final String DEFAULT_STRATEGY_NAME = "default";

    @Autowired
    public AdvertStrategyServiceImpl(ActivityProperties activityProperties) {
        this.activityProperties = activityProperties;
    }

    @Override
    public List<AdvertStrategyDTO> getAdvertStrategy(BasicParam basicParam) {
        //每次解析间隔获取一次远程的配置
        if (System.currentTimeMillis() - lastParseTime > PARSE_INTERVAL) {
            parse();
        }

        List<AdvertStrategyDTO> matchRes = cacheMap.get(basicParam.getCurVer());

        if (null == matchRes) {
            matchRes = defaultMatch;
        }

        return matchRes;
    }

    /**
     * 解析配置的数据，方便获取
     */
    private void parse() {
        String config = activityProperties.getAdvertStrategyConfigList();
        defaultMatch = null;
        cacheMap = Maps.newHashMap();

        if (StringUtils.isNotBlank(config)) {
            List<AdvertStrategyConfigDTO> advertStrategyConfigList = JSONArray.parseArray(config, AdvertStrategyConfigDTO.class);

            int configSize = advertStrategyConfigList.size();

            if (configSize > 0) {
                advertStrategyConfigList.forEach(strategy -> cacheMap.put(strategy.getVersion(), strategy.getConfig()));

                //获取默认的配置，如果未配置则将最后一条作为配置返回
                if (cacheMap.containsKey(DEFAULT_STRATEGY_NAME)) {
                    defaultMatch = cacheMap.get(DEFAULT_STRATEGY_NAME);
                } else {
                    defaultMatch = advertStrategyConfigList.get(configSize - 1).getConfig();
                }
            }

        }

        lastParseTime = System.currentTimeMillis();
    }
}
