package com.bxm.fossicker.activity.service.impl;

import com.bxm.fossicker.activity.model.dto.FriendsInviteInfoDTO;
import com.bxm.fossicker.activity.model.dto.FriendsInviteProgressDTO;
import com.bxm.fossicker.activity.model.dto.InviteUserBaseInfoDTO;
import com.bxm.fossicker.activity.model.dto.TotalFriendsInfoDTO;
import com.bxm.fossicker.activity.model.param.FriendsListParam;
import com.bxm.fossicker.activity.model.vo.UserInviteRelation;
import com.bxm.fossicker.activity.service.invite.InviteRelationService;
import com.bxm.fossicker.activity.service.config.ActivityProperties;
import com.bxm.fossicker.activity.service.friends.FriendsInfoService;
import com.bxm.fossicker.user.facade.CashFacadeService;
import com.bxm.fossicker.user.facade.GoldFacadeService;
import com.bxm.fossicker.user.facade.UserInfoFacadeService;
import com.bxm.fossicker.user.facade.dto.UserInfoDto;
import com.bxm.fossicker.user.facade.enums.VipTypeEnum;
import com.bxm.fossicker.user.facade.vip.VipFacadeService;
import com.bxm.fossicker.user.model.dto.CashFlowDto;
import com.bxm.fossicker.user.model.dto.GoldFlowDto;
import com.bxm.fossicker.user.model.entity.UserGoldFlowBean;
import com.bxm.fossicker.user.model.param.FlowPageParam;
import com.bxm.fossicker.vo.PageWarper;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.StringUtils;
import com.github.pagehelper.Page;
import com.github.pagehelper.PageHelper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

@Service
public class FriendsInfoServiceImpl implements FriendsInfoService {

    @Autowired
    @Lazy
    private InviteRelationService inviteRelationService;

    @Autowired
    private ActivityProperties activityProperties;

    @Autowired
    private VipFacadeService vipFacadeService;

    @Autowired
    private UserInfoFacadeService userInfoFacadeService;

    @Autowired
    private GoldFacadeService goldFacadeService;

    @Autowired
    private CashFacadeService cashFacadeService;

    @Override
    public TotalFriendsInfoDTO getTotalFriendsInfo(Long userId) {
        //获取好友总数
        Integer totalFriends = inviteRelationService.getFriendsNum(userId);
        //获取收益
        BigDecimal earning = inviteRelationService.getEarning(userId);

        return TotalFriendsInfoDTO.builder()
                .friendNum(totalFriends)
                .earnings(Objects.isNull(earning) ? 0D : earning.doubleValue())
                //总收益为好友总数 * 25
                .totalEarnings(totalFriends * 25D)
                .build();
    }

    @Override
    public FriendsInviteProgressDTO getInvitation(Long userId) {

        //现金收益版本好友邀请下的邀请好友总数
        Integer friendsNum = inviteRelationService.getFriendsNumByVersion(userId);
        if (null == friendsNum || 0 == friendsNum) {
            return null;
        }

        return FriendsInviteProgressDTO.builder()
                .friendNum(friendsNum)
                //获取邀请得到的vip卡数
                .vipCardNum(vipFacadeService.getVipCardByInvite(userId))
                .userProcessInfo(getUserProcessInfo(friendsNum, userId))
                .build();
    }

    @Override
    public PageWarper<FriendsInviteInfoDTO> getFriendsList(FriendsListParam friendsListParam) {

        PageWarper<FriendsInviteInfoDTO> resultPage = new PageWarper<>();
        resultPage.setPageNum(friendsListParam.getPageNum());
        resultPage.setPageSize(friendsListParam.getPageSize());

        //获取所有好友列表
        Page<UserInviteRelation> friendsListPage = inviteRelationService.getAllFriendsList(friendsListParam);
        if (CollectionUtils.isEmpty(friendsListPage.getResult())) {
            return resultPage;
        }
        // 填充用户的信息
        Map<Long, UserInfoDto> userInfoDtoMap = fillUserExtInfo(friendsListPage.getResult());

        List<FriendsInviteInfoDTO> resultList =
                friendsListPage.stream().map(e-> convertAllFriends(e,userInfoDtoMap)).collect(Collectors.toList());
        resultPage.setList(resultList);
        return resultPage;
    }

    private Map<Long, UserInfoDto> fillUserExtInfo(List<UserInviteRelation> resultList) {
        List<UserInfoDto> userInfoDtoList
                = userInfoFacadeService.getUserByIds(resultList.stream().map(UserInviteRelation::getInvitedId).collect(Collectors.toList()));
        Map<Long, UserInfoDto> userInfoDtoMap = userInfoDtoList.stream().collect(Collectors.toMap(UserInfoDto::getId, Function.identity(), (key1, key2) -> key2));
        resultList.forEach(e -> {
            UserInfoDto userInfoDto = userInfoDtoMap.get(e.getInvitedId());
            if (Objects.nonNull(userInfoDto)) {
                if (StringUtils.isNotBlank(userInfoDto.getNickName())) {
                    e.setInvitedName(userInfoDto.getNickName());
                }
                if (StringUtils.isNotBlank(userInfoDto.getHeadImg())) {
                    e.setInvitedHeadImg(userInfoDto.getHeadImg());
                }
            }
        });
        return userInfoDtoMap;
    }

    /**
     * 获取用户进度
     * @param friendsNum
     * @param userId
     * @return
     */
    private List<InviteUserBaseInfoDTO> getUserProcessInfo(Integer friendsNum, Long userId) {
        List<InviteUserBaseInfoDTO> resultList = new ArrayList<>();

        if (null == friendsNum || 0 == friendsNum) {
            return resultList;
        }
        //获取好友数取余奖励的人数,能得到最近的最多2位的用户,可能值0/1/2
        int validNum = friendsNum % activityProperties.getInviteUserRewardNum();
        if (0 == validNum) {
            return resultList;
        }

        List<UserInviteRelation> friendsList = inviteRelationService.getFriendsList(userId);
        if (CollectionUtils.isEmpty(friendsList)) {
            return resultList;
        }

        // 截取数组-按邀请成功时间倒序排序-数据转换
        return friendsList.subList(0, validNum).stream()
                .sorted(Comparator.comparing(UserInviteRelation::getInviteSuccessTime))
                .map(this::convert)
                .collect(Collectors.toList());
    }

    private InviteUserBaseInfoDTO convert(UserInviteRelation userInviteRelation) {

        InviteUserBaseInfoDTO resultInfo =  InviteUserBaseInfoDTO.builder()
                .userId(userInviteRelation.getInvitedId())
                .headImg(userInviteRelation.getInvitedHeadImg())
                .build();

        // 获取最新的用户头像
        UserInfoDto invitedUserInfo = userInfoFacadeService.getUserById(userInviteRelation.getInvitedId());
        if (Objects.nonNull(invitedUserInfo)) {
            if (StringUtils.isNotBlank(invitedUserInfo.getHeadImg())) {
                resultInfo.setHeadImg(invitedUserInfo.getHeadImg());
            }
        }

        return resultInfo;
    }

    private FriendsInviteInfoDTO convertAllFriends(UserInviteRelation userInviteRelation, Map<Long, UserInfoDto> userInfoDtoMap) {

        FriendsInviteInfoDTO friendsInviteInfoDTO = FriendsInviteInfoDTO.builder()
                .headImg(userInviteRelation.getInvitedHeadImg())
                .goldEarnings(userInviteRelation.getGoldEarnings())
                .inviteType(userInviteRelation.getInviteType())
                .nickName(userInviteRelation.getInvitedName())
                .phoneNum(hidePhoneInfo(userInviteRelation.getInvitedPhone()))
                .earnings(userInviteRelation.getInviteEarning().doubleValue())
                .inviteSuccessTime(DateUtils.formatAtWill(userInviteRelation.getInviteSuccessTime(), "yyyy/MM/dd"))
                // 1.4.2改为淘宝、微信绑定成功
                .loginSuccessFlag(false)
                //invite的状态判断有问题
                .inviteSuccessFlag(1 == userInviteRelation.getStatus() || (0 == userInviteRelation.getStatus()))
                .build();

        //获取最新的昵称和头像
        UserInfoDto invitedUserInfo = userInfoDtoMap.get(userInviteRelation.getInvitedId());
        if (Objects.nonNull(invitedUserInfo)) {
            friendsInviteInfoDTO.setUserId(userInviteRelation.getInvitedId());
            if (StringUtils.isNotBlank(invitedUserInfo.getHeadImg())) {
                friendsInviteInfoDTO.setHeadImg(invitedUserInfo.getHeadImg());
            }
            if (StringUtils.isNotBlank(invitedUserInfo.getNickName())) {
                friendsInviteInfoDTO.setNickName(invitedUserInfo.getNickName());
            }

            // 淘宝、微信绑定成功
            friendsInviteInfoDTO.setLoginSuccessFlag(StringUtils.isNotBlank(invitedUserInfo.getWechatNickName())
                    && StringUtils.isNotBlank(invitedUserInfo.getTaobaoNickName()));

            //判断是否开通vip，显示不同的文案
            if (VipTypeEnum.isUsable((byte) invitedUserInfo.getVip())) {
                friendsInviteInfoDTO.setDesc("提醒好友赚现金，享10%好友收入分成");
            }else{
                friendsInviteInfoDTO.setDesc("提醒好友开通黑钻卡，得5元现金奖励");
            }
        }

        Integer vipCardNum = vipFacadeService.getTotalVipCard(userInviteRelation.getInvitedId());
        if (Objects.isNull(vipCardNum) || 0 == vipCardNum) {
            friendsInviteInfoDTO.setVipSuccessFlag(false);
            return friendsInviteInfoDTO;
        }
        friendsInviteInfoDTO.setVipSuccessFlag(true);
        friendsInviteInfoDTO.setVipCardNum(vipCardNum);


        return friendsInviteInfoDTO;
    }

    /**
     * 手机号隐藏信息
     * @param phone 手机号
     * @return 隐藏后的信息
     */
    private String hidePhoneInfo(String phone) {
        if (StringUtils.isBlank(phone)) {
            return phone;
        }

        return phone.replaceAll("(\\d{3})\\d{4}(\\d{4})", "$1****$2");
    }

    @Override
    public List<GoldFlowDto> getFriendGoldFlow(FlowPageParam param) {
        return goldFacadeService.getFriendGoldFlow(param);
    }

    @Override
    public List<CashFlowDto> getFriendCashFlow(FlowPageParam param) {
        return cashFacadeService.getFriendCashFlow(param);
    }
}
