package com.bxm.fossicker.activity.service.impl;


import com.bxm.fossicker.activity.config.InviteConfig;
import com.bxm.fossicker.activity.domain.UserAddressListMapper;
import com.bxm.fossicker.activity.model.dto.PhoneStatusDto;
import com.bxm.fossicker.activity.model.param.AddPhoneListParam;
import com.bxm.fossicker.activity.model.param.AddPhoneParam;
import com.bxm.fossicker.activity.model.vo.UserAddressList;
import com.bxm.fossicker.activity.service.UserAddressListService;
import com.bxm.newidea.component.uuid.SequenceCreater;
import lombok.extern.log4j.Log4j2;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @ClassName UserAddressListServiceImpl
 * @CopyRright (c) 2019-bxm：杭州微财网络科技有限公司
 * @Author XinZhao
 * @Date 2019/7/2 17:49
 * @Version 1.0.0
 **/
@Log4j2
@Service
public class UserAddressListServiceImpl implements UserAddressListService {

    private final UserAddressListMapper addressListMapper;

    private final SequenceCreater sequenceCreater;

    private final InviteConfig inviteConfig;

    @Autowired
    public UserAddressListServiceImpl(UserAddressListMapper addressListMapper, SequenceCreater sequenceCreater, InviteConfig inviteConfig) {
        this.addressListMapper = addressListMapper;
        this.sequenceCreater = sequenceCreater;
        this.inviteConfig = inviteConfig;
    }

    @Override
    @Transactional
    public List<PhoneStatusDto> uploadPhone(AddPhoneListParam addPhoneListParam) {
        if (log.isDebugEnabled()) {
            log.debug("用户开始上传通讯录号码,[{}]", addPhoneListParam.toString());
        }

        List<UserAddressList> existUserAddressLists = addressListMapper.listByUserId(addPhoneListParam.getUserId());

        if (!CollectionUtils.isEmpty(addPhoneListParam.getAddPhoneList())) {

            // 手机号 分隔符清理
            cleanPhones(addPhoneListParam);

            // 该用户已上传的通讯录
            Set<String> phoneSet = existUserAddressLists.stream()
                    .map(UserAddressList::getPhone)
                    .collect(Collectors.toSet());

            // 过滤后手机号列表
            List<UserAddressList> newUserAddressLists = addPhoneListParam.getAddPhoneList().stream()
                    .filter(addPhoneParam -> StringUtils.isNotBlank(addPhoneParam.getPhone())
                            && addPhoneParam.getPhone().trim().length() == 11  // 长度限制
                            && addPhoneParam.getPhone().trim().startsWith("1")
                            && !phoneSet.contains(addPhoneParam.getPhone())) // 去重
                    .map(addPhoneParam -> build(addPhoneListParam.getUserId(), addPhoneParam))
                    .collect(Collectors.toList());

            if (!CollectionUtils.isEmpty(newUserAddressLists)) {
                addressListMapper.batchAdd(newUserAddressLists);
                // 拼接手机号列表
                existUserAddressLists.addAll(newUserAddressLists);
            }
        }


        return existUserAddressLists.stream().map(this::convert).collect(Collectors.toList());
    }

    private void cleanPhones(AddPhoneListParam addPhoneListParam) {
        addPhoneListParam.getAddPhoneList().forEach(addPhoneParam -> {
            String phone = addPhoneParam.getPhone();
            if (StringUtils.isNotBlank(phone)){
                if (phone.contains("-")){
                     phone = phone.replace("-", "");
                }
                if (phone.contains("+86")){
                     phone = phone.replace("+86", "");
                }
                if (phone.contains(" ")){
                    phone = phone.replace(" ", "");
                }
                addPhoneParam.setPhone(phone);
            }
        });
    }

    private UserAddressList build(Long userId, AddPhoneParam addPhoneParam) {
        return UserAddressList.builder()
                .id(sequenceCreater.nextLongId())
                .inviteStatus(0)
                .linkman(addPhoneParam.getName())
                .phone(addPhoneParam.getPhone())
                .userId(userId)
                .build();
    }

    @Override
    @Transactional
    public Boolean invited(Long userId, String phone) {
        Long id = exists(userId, phone);
        return null == id ? add(userId, phone) : invited(id);
    }

    @Override
    public Long exists(Long userId, String phone) {
        UserAddressList addressList = addressListMapper.query(userId, phone);
        return null == addressList ? null : addressList.getId();
    }

    @Override
    public Boolean invited(Long id) {
        return addressListMapper.updateStatus(id, 1) > 0;
    }

    @Override
    public Boolean add(Long userId, String phone) {
        return addressListMapper.addOne(
                UserAddressList.builder()
                        .id(sequenceCreater.nextLongId())
                        .userId(userId)
                        .phone(phone)
                        .linkman("")
                        .inviteStatus(1)
                        .build());
    }

    private PhoneStatusDto convert(UserAddressList userAddressList) {
        return PhoneStatusDto.builder()
                .id(userAddressList.getId())
                .name(userAddressList.getLinkman())
                .phone(new StringBuffer(userAddressList.getPhone()).replace(3, 7, "****").toString())
                .invited(userAddressList.getInviteStatus() == 1)
                .probablyAmount(inviteConfig.getReward().toString())
                .build();
    }
}
