package com.bxm.fossicker.activity.service.task.action.impl;

import com.bxm.fossicker.activity.constants.TaskConstant;
import com.bxm.fossicker.activity.domain.ActivityTaskMapper;
import com.bxm.fossicker.activity.enums.DisplayEnum;
import com.bxm.fossicker.activity.enums.TaskStatusEnum;
import com.bxm.fossicker.activity.model.vo.ActivityUserTask;
import com.bxm.fossicker.activity.service.task.NewUserActivityTaskService;
import com.bxm.fossicker.activity.service.task.action.ActivityTaskAction;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.uuid.SequenceCreater;
import lombok.extern.log4j.Log4j2;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Date;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

/**
 * 活动_任务_新手任务实现
 *
 * @author xin.zhao
 */
@Service("new_user_task")
@Log4j2
public class NewUserActivityTaskActionImpl implements ActivityTaskAction {

    @Autowired
    private  ActivityTaskMapper taskMapper;

    @Autowired
    private SequenceCreater sequenceCreater;

    @Autowired
    private NewUserActivityTaskService newUserActivityTaskService;

    @Autowired
    private DistributedLock distributedLock;


    public static final String COMPLETE = "COMPLETE";

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void complete(Long userId, String taskCode) {

        String key = StringUtils.join(taskCode, COMPLETE, Objects.toString(userId));
        // 避免并发
        if (!distributedLock.lock(key, userId.toString(), 2, TimeUnit.SECONDS)) {
            log.warn("完成任务分布式锁获取失败, userId: {} taskCode: {}", userId, taskCode);
            return;
        }

        Optional<ActivityUserTask> userTaskOptional = Optional.ofNullable(
                taskMapper.query(userId, taskCode)
        );

        if (userTaskOptional.isPresent()) {
            // 存在则可能是1.4.1版本前用户，需要校验任务状态
            if (userTaskOptional.get().getStatus().equals(TaskStatusEnum.UNFINISHED.getCode())) {
                updateComplete(userTaskOptional.get().getId());
            }
        } else {
            // 不存在则是1.4.1版本新用户，添加完成状态
            addComplete(userId, taskCode);
        }

        distributedLock.unlock(key, userId.toString());
    }

    /**
     * 自 1.4.1版本后新手任务完成则添加一条完成记录
     *
     * @param userId   用户id
     * @param taskCode 任务id
     */
    private void addComplete(Long userId, String taskCode) {
        taskMapper.addOne(
                ActivityUserTask.builder()
                        .id(sequenceCreater.nextLongId())
                        .taskCode(taskCode)
                        .status(TaskStatusEnum.FINISH_UNOBTAIN.getCode())
                        .residueTime(TaskConstant.DEFAULT_RESIDUE_TIME)
                        .display(DisplayEnum.SHOW.getCode())
                        .userId(userId)
                        .createTime(new Date())
                        .build()
        );
    }

    /**
     * 1.4.1版本之前 用户完成任务为更新任务状态
     *
     * @param userTaskId 用户任务id
     */
    private void updateComplete(Long userTaskId) {
        // 更新任务状态为1：已完成未领取
        taskMapper.completeById(userTaskId);
    }

    @Override
    public Boolean allowReward(Long userId, String taskCode) {
        // 更新新手任务为已领取
        return newUserActivityTaskService.updateReward(userId, taskCode);
    }

}
